package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to move
 * devices to manager (even from other company) through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.devices().move(IDs)
 *      .toManager(managerId)
 *      .toCompany(companyId)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that at least manager must be specified so invocation of {@code toManager} method
 * is mandatory. Otherwise an exception will be thrown.
 */
public class MoveDeviceRequestExecutor extends RequestExecutor<String> {

  private final DevicesService devicesService;

  private final String[] uniqueIds;
  private String managerEmail;
  private UUID companyId;

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   *
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public MoveDeviceRequestExecutor(final DevicesService devicesService, final String... uniqueIds) {
    this.devicesService = devicesService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   *
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public MoveDeviceRequestExecutor(final DevicesService devicesService, final List<String> uniqueIds) {
    this.devicesService = devicesService;
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
  }

  /**
   * Specifies the manager. The method invocation is obligatory while using this request executor.
   *
   * @param managerEmail manager email address.
   * @return this request executor.
   */
  public MoveDeviceRequestExecutor toManager(final String managerEmail) {
    SDKPreconditions.checkNotNull(managerEmail, "manager email cannot be null");
    this.managerEmail = managerEmail;
    return this;
  }

  /**
   * Specifies the company.
   *
   * @param companyId company unique identifier.
   * @return this request executor.
   */
  public MoveDeviceRequestExecutor toCompany(final UUID companyId) {
    SDKPreconditions.checkNotNull(companyId, "company ID cannot be null");
    this.companyId = companyId;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return devicesService.moveDevice(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(managerEmail != null, "cannot move beacon - specify manager");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    params.put(CloudConstants.Devices.MANAGER_EMAIL_PARAMETER, managerEmail);
    if (companyId != null) {
      params.put(CloudConstants.Devices.COMPANY_ID_PARAMETER, companyId.toString());
    }
    return params;
  }
}
