package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.response.paginated.CredentialsList;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to get
 * device credentials list through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   CredentialsList credentialsList = kontaktCloud.devices().credentials(deviceIDs).execute();
 *   </code>
 * </pre>
 */
public class CredentialsListRequestExecutor extends PaginatedRequestExecutor<CredentialsList> {

  private final DevicesService devicesService;

  private final List<String> uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   *
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public CredentialsListRequestExecutor(final DevicesService devicesService, final List<String> uniqueIds) {
    this.devicesService = devicesService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<CredentialsList> prepareCall() {
    return devicesService.getDeviceCredentialsByUniqueIds(StringUtils.join(uniqueIds, ","), params());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CredentialsListRequestExecutor startIndex(final int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CredentialsListRequestExecutor maxResult(final int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CredentialsListRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CredentialsListRequestExecutor filter(final String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CredentialsListRequestExecutor eTag(final String eTag) {
    super.eTag(eTag);
    return this;
  }

  @Override
  protected Map<String, String> params() {
    return super.params();
  }
}
