package com.kontakt.sdk.android.cloud.api.executor.actions;

import android.util.Base64;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActionsApi;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.UUID;

/**
 * Request executor provided by {@link ActionsApi}. Use this class if you want to update
 * content actions through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   KontaktCloud kontaktCloud = KontaktCloudFactory.create();
 *   kontaktCloud.actions().updateContentAction(id)
 *      .withProximity(Proximity.IMMEDIATE)
 *      .withMediaFile(new File(fileName))
 *      .execute();
 *   </code>
 * </pre>
 */
public class UpdateContentActionRequestExecutor extends UpdateActionRequestExecutor {

  private String encodedFile;

  /**
   * Constructs request executor initialized with corresponding service class and action's id.
   *
   * @param actionsService teh actions API service.
   * @param id the action's unique identifier.
   */
  public UpdateContentActionRequestExecutor(final ActionsService actionsService, final UUID id) {
    super(actionsService, id);
  }

  /**
   * Updates the action's media file.
   *
   * @param file the media file.
   * @return this request executor.
   * @throws IOException instance of IOException
   */
  public UpdateContentActionRequestExecutor withMediaFile(final File file) throws IOException {
    SDKPreconditions.checkNotNull(file, "file is null");
    SDKPreconditions.checkState(file.exists(), "file does not exist");
    final byte[] bytes = ConversionUtils.convert(file);
    this.encodedFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UpdateContentActionRequestExecutor withProximity(Proximity proximity) {
    super.withProximity(proximity);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void checkPreconditions() {
    SDKPreconditions.checkState(encodedFile != null, "cannot update action - specify media file");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    params.put(CloudConstants.Actions.FILE_PARAMETER, encodedFile);

    return params;
  }
}
