package com.kontakt.sdk.android.cloud.api;

import com.kontakt.sdk.android.cloud.api.executor.actions.ActionsRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.actions.ContentActionFileRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.actions.CreateBrowserActionRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.actions.CreateContentActionRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.actions.DeleteActionRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.actions.UpdateBrowserActionRequestExecutor;
import com.kontakt.sdk.android.cloud.api.executor.actions.UpdateContentActionRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActionsService;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.UUID;

/**
 * An abstract representation of Actions API endpoint. Provides executors that can be
 * used to handle all action-related requests.
 */
public class ActionsApi {

  private final ActionsService actionsService;

  /**
   * Constructs Actions API endpoint initialized with corresponding service class.
   *
   * @param actionsService the actions API service.
   */
  public ActionsApi(final ActionsService actionsService) {
    this.actionsService = actionsService;
  }

  /**
   * Provides executor that can be used to obtain actions.
   *
   * @return the request executor.
   */
  public ActionsRequestExecutor fetch() {
    return new ActionsRequestExecutor(actionsService);
  }

  /**
   * Provides executor that can be used to load content action file.
   *
   * @param id the action's unique identifier.
   * @return the request executor.
   */
  public ContentActionFileRequestExecutor fileOf(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new ContentActionFileRequestExecutor(actionsService, id);
  }

  /**
   * Provides executor that can be used to create a content action.
   *
   * @return the request executor.
   */
  public CreateContentActionRequestExecutor createContentAction() {
    return new CreateContentActionRequestExecutor(actionsService);
  }

  /**
   * Provides executor that can be used to create a browser action.
   *
   * @return the request executor.
   */
  public CreateBrowserActionRequestExecutor createBrowserAction() {
    return new CreateBrowserActionRequestExecutor(actionsService);
  }

  /**
   * Provides executor that can be used to update a content action.
   *
   * @param id the action's unique identifier.
   * @return the request executor.
   */
  public UpdateContentActionRequestExecutor updateContentAction(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new UpdateContentActionRequestExecutor(actionsService, id);
  }

  /**
   * Provides executor that can be used to update a browser action.
   *
   * @param id the action's unique identifier.
   * @return the request executor.
   */
  public UpdateBrowserActionRequestExecutor updateBrowserAction(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new UpdateBrowserActionRequestExecutor(actionsService, id);
  }

  /**
   * Provides executor that can be used to delete an action.
   *
   * @param id the action's unique identifier.
   * @return the request executor.
   */
  public DeleteActionRequestExecutor delete(final UUID id) {
    SDKPreconditions.checkNotNull(id, "ID cannot be null");
    return new DeleteActionRequestExecutor(actionsService, id);
  }
}
