package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanSettings;
import android.os.Build;
import com.kontakt.sdk.android.ble.configuration.ForceScanConfiguration;
import com.kontakt.sdk.android.common.log.Logger;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

final class RunnersL {

  private RunnersL() {
  }

  public static Runnable newRunner(final Runners.RunnerType runnerType, final ScanConfiguration configuration) {
    switch (runnerType) {
      case MONITOR_ACTIVE_RUNNER:
        return newMonitorActiveRunnerL((ScanConfigurationL) configuration);
      case MONITOR_PASSIVE_RUNNER:
        return newMonitorPassiveRunnerL((ScanConfigurationL) configuration);
      case FORCE_SCAN_RUNNER:
        return newForceScanRunnerL((ScanConfigurationL) configuration);
      default:
        throw new RuntimeException();
    }
  }

  private static Runnable newMonitorPassiveRunnerL(final ScanConfigurationL configuration) {
    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.LOLLIPOP)
      @Override
      public void run() {
        final BluetoothLeScanner scanner = getScanner();

        Logger.d("Stopping scan");
        scanner.stopScan(monitorCallback);

        Logger.d("On monitor stopped");
        monitorCallback.onMonitorStopped();
      }
    };
  }

  private static Runnable newMonitorActiveRunnerL(final ScanConfigurationL configuration) {
    final MonitorCallbackL monitorCallback = (MonitorCallbackL) configuration.getScanCallback();
    final ScanSettings scanSettings = configuration.getScanSettings();
    final List<ScanFilter> scanFilterList = new ArrayList<>();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.LOLLIPOP)
      @Override
      public void run() {
        final BluetoothLeScanner scanner = getScanner();
        Logger.d("Start Scan");

        scanner.startScan(scanFilterList, scanSettings, monitorCallback);

        Logger.d("On Monitor started");
        monitorCallback.onMonitorStarted();
      }
    };
  }

  private static Runnable newForceScanRunnerL(final ScanConfigurationL configuration) {
    final MonitorCallbackL scanCallback = (MonitorCallbackL) configuration.getScanCallback();
    final ScanSettings scanSettings = configuration.getScanSettings();

    final ForceScanConfiguration forceScanConfiguration = configuration.getScanContext().getForceScanConfiguration();

    final long forceScanActivePeriod = forceScanConfiguration.getForceScanActivePeriod();
    final long forceScanPassivePeriod = forceScanConfiguration.getForceScanPassivePeriod();

    final List<ScanFilter> scanFilterList = new ArrayList<>();

    return new Runnable() {
      @TargetApi(Build.VERSION_CODES.LOLLIPOP)
      @Override
      public void run() {
        final BluetoothLeScanner scanner = getScanner();
        try {
          if (scanner == null) {
            Logger.e("Unexpected error occured - Bluetooth Scanner is null");
          }

          while (!Thread.currentThread().isInterrupted()) {
            Logger.d("Stopping Scan (force)");
            scanner.stopScan(scanCallback);

            Logger.d(String.format("Sleep during passive period: %s", String.valueOf(forceScanPassivePeriod)));
            TimeUnit.MILLISECONDS.sleep(forceScanPassivePeriod);

            Logger.d("Starting scan (force)");
            scanner.startScan(scanFilterList, scanSettings, scanCallback);

            Logger.d(String.format("Sleep during active period: %s", String.valueOf(forceScanActivePeriod)));
            TimeUnit.MILLISECONDS.sleep(forceScanActivePeriod);
          }

          scanner.stopScan(scanCallback);
          Logger.d("Force scan finished");
        } catch (InterruptedException e) {
          scanner.stopScan(scanCallback);
          Logger.d("Force scan interrupted");
        }
      }
    };
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP)
  static BluetoothLeScanner getScanner() {
    final BluetoothAdapter adapter = BluetoothAdapter.getDefaultAdapter();
    if (adapter == null) {
      return null;
    }
    return adapter.getBluetoothLeScanner();
  }
}
