package com.kontakt.sdk.android.ble.security.auth;

import com.kontakt.sdk.android.ble.security.EncryptedReadAllRequest;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.KontaktCloudFactory;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.CloudError;
import com.kontakt.sdk.android.cloud.response.CloudHeaders;
import com.kontakt.sdk.android.cloud.response.paginated.Configs;
import com.kontakt.sdk.android.common.model.Credentials;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * AuthToken is used to authorize with beacons with firmwares 4.0 or higher and beacons pro.
 * <br>
 * This authorization is required before conducting some of the operations.
 * <br><br>
 * AuthToken can be obtained in 2 ways. Using Kontakt Cloud (recommended!) and offline - by providing device's password.
 * The latter is not recommended as it may desynchronize beacon with the cloud and should only be used if one knows what he is doing.
 */
public class AuthToken {

  private final String uniqueId;
  private final String password;
  private final String token;

  /**
   * Used to obtain AuthToken by requesting it from Kontakt Cloud.
   * <br>
   * This is recommended and safe way of obtaining AuthTokens.
   * <br><br>
   * This method creates {@link KontaktCloud} instance implicitly.
   * If you already have an instance you can use {@link #obtain(String, KontaktCloud, AuthTokenCallback)} method.
   *
   * @param uniqueId Beacon's unique id
   * @param callback {@link AuthTokenCallback} instance.
   */
  public static void obtain(final String uniqueId, final AuthTokenCallback callback) {
    obtain(uniqueId, KontaktCloudFactory.create(), callback);
  }

  /**
   * Used to obtain AuthToken by requesting it from Kontakt Cloud.
   * <br>
   * This is recommended and safe way of obtaining AuthTokens.
   *
   * @param uniqueId Beacon's unique id
   * @param callback {@link AuthTokenCallback} instance.
   */
  public static void obtain(final String uniqueId, final KontaktCloud kontaktCloud, final AuthTokenCallback callback) {
    checkNotNull(uniqueId, "Unique ID can't be null");
    checkNotNull(kontaktCloud, "KontaktCloud can't be null");
    checkNotNull(callback, "Callback can't be null");

    kontaktCloud.devices().credentials(uniqueId).execute(new CloudCallback<Credentials>() {
      @Override
      public void onSuccess(Credentials response, CloudHeaders headers) {
        final String password = response.getPassword();
        kontaktCloud.configs().readAll().withIds(uniqueId).execute(new CloudCallback<Configs>() {
          @Override
          public void onSuccess(Configs response, CloudHeaders headers) {
            String token = response.getContent().get(0).getSecureRequest();
            callback.onSuccess(new AuthToken(uniqueId, password, token));
          }

          @Override
          public void onError(CloudError error) {
            callback.onError(error.getMessage());
          }
        });
      }

      @Override
      public void onError(CloudError error) {
        callback.onError(error.getMessage());
      }
    });
  }

  /**
   * Used to obtain AuthToken by providing beacon's password.
   * <br>
   * WARNING! This should only be used in special cases as offline authorization may desynchronize the beacon with Kontakt Cloud.
   * Always obtain AuthToken by communicating with KontaktCloud when possible.
   *
   * @param password Beacon's password
   * @return AuthToken object
   */
  public static AuthToken obtain(final String password) {
    checkNotNull(password, "Password can't be null");

    int timestampToken = (int) (System.currentTimeMillis() / 1000L);
    EncryptedReadAllRequest readAllRequest = new EncryptedReadAllRequest(timestampToken, password);
    return new AuthToken("", password, readAllRequest.getBase64Data());
  }

  AuthToken(String uniqueId, String password, String token) {
    this.uniqueId = uniqueId;
    this.token = token;
    this.password = password;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  public String getToken() {
    return token;
  }

  public String getPassword() {
    return password;
  }

  /**
   * Used to provide AuthToken when requesting from Kontakt Cloud.
   */
  public interface AuthTokenCallback {
    void onSuccess(AuthToken token);

    void onError(String error);
  }
}
