package com.kontakt.sdk.android.ble.manager;

import android.content.Context;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.manager.configuration.FiltersConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.GeneralConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.SpacesConfigurator;
import com.kontakt.sdk.android.ble.manager.internal.InternalProximityManager;
import com.kontakt.sdk.android.ble.manager.listeners.EddystoneListener;
import com.kontakt.sdk.android.ble.manager.listeners.IBeaconListener;
import com.kontakt.sdk.android.ble.manager.listeners.ScanStatusListener;
import com.kontakt.sdk.android.ble.manager.listeners.SecureProfileListener;
import com.kontakt.sdk.android.ble.manager.listeners.SpaceListener;
import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.KontaktCloudFactory;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.DeviceProfile;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class ProximityManagerImpl implements ProximityManager {

  private IBeaconListener iBeaconListener;
  private EddystoneListener eddystoneListener;
  private SecureProfileListener secureProfileListener;
  private ScanStatusListener scanStatusListener;
  private SpaceListener spaceListener;
  private InternalProximityManager internalProximityManager;
  private ScanContext.Builder scanContext = new ScanContext.Builder();

  ProximityManagerImpl(Context context) {
    this(context, KontaktCloudFactory.create());
  }

  private ProximityManagerImpl(Context context, KontaktCloud kontaktCloud) {
    checkNotNull(context, "Context can't be null").getApplicationContext();
    checkNotNull(kontaktCloud, "Kontakt Cloud can't be null");
    this.internalProximityManager = new InternalProximityManager(context.getApplicationContext(), kontaktCloud);
  }

  @Override
  public void connect(OnServiceReadyListener onServiceReadyListener) {
    checkNotNull(onServiceReadyListener, "OnServiceReadyListener can't be null.");
    if (!internalProximityManager.isConnected()) {
      internalProximityManager.connect(onServiceReadyListener);
    } else {
      onServiceReadyListener.onServiceReady();
      Logger.d("ProximityManager is already connected.");
    }
  }

  @Override
  public void disconnect() {
    internalProximityManager.disconnect();
  }

  @Override
  public void startScanning() {
    checkArgument(internalProximityManager.isConnected(),
        "ProximityManager is not connected to ProximityService. Use ProximityManager.connect() before starting a scan.");
    if (!internalProximityManager.isScanning()) {
      checkObservedProfiles();
      internalProximityManager.initializeScan(scanContext.build(), createEventObserver());
    }
  }

  @Override
  public void restartScanning() {
    checkArgument(internalProximityManager.isConnected(),
        "ProximityManager is not connected to ProximityService. Use ProximityManager.connect() before starting a scan.");
    if (internalProximityManager.isScanning()) {
      checkObservedProfiles();
      internalProximityManager.restartScan(scanContext.build(), createEventObserver());
    }
  }

  @Override
  public void stopScanning() {
    if (internalProximityManager.isScanning()) {
      internalProximityManager.finishScan();
    }
  }

  @Override
  public boolean isConnected() {
    return internalProximityManager.isConnected();
  }

  @Override
  public boolean isScanning() {
    return internalProximityManager.isScanning();
  }

  @Override
  public void setScanStatusListener(ScanStatusListener listener) {
    this.scanStatusListener = listener;
  }

  @Override
  public void setSpaceListener(SpaceListener listener) {
    this.spaceListener = listener;
  }

  @Override
  public void setIBeaconListener(IBeaconListener listener) {
    this.iBeaconListener = listener;
  }

  @Override
  public void setSecureProfileListener(SecureProfileListener listener) {
    this.secureProfileListener = listener;
  }

  @Override
  public void setEddystoneListener(EddystoneListener listener) {
    this.eddystoneListener = listener;
  }

  @Override
  public GeneralConfigurator configuration() {
    return scanContext;
  }

  @Override
  public SpacesConfigurator spaces() {
    return scanContext;
  }

  @Override
  public FiltersConfigurator filters() {
    return scanContext;
  }

  private void checkObservedProfiles() {
    if (iBeaconListener == null) {
      scanContext.removeObservedProfile(DeviceProfile.IBEACON);
    } else {
      scanContext.addObservedProfile(DeviceProfile.IBEACON);
    }

    if (eddystoneListener == null) {
      scanContext.removeObservedProfile(DeviceProfile.EDDYSTONE);
    } else {
      scanContext.addObservedProfile(DeviceProfile.EDDYSTONE);
    }

    if (secureProfileListener == null) {
      scanContext.removeObservedProfile(DeviceProfile.KONTAKT_SECURE);
    } else {
      scanContext.addObservedProfile(DeviceProfile.KONTAKT_SECURE);
    }
  }

  private EventObserver createEventObserver() {
    return new EventObserver(eddystoneListener, iBeaconListener, scanStatusListener, spaceListener, secureProfileListener);
  }
}
