package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.bluetooth.BluetoothDevice;
import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.EddystoneNamespace;
import com.kontakt.sdk.android.ble.discovery.AbstractDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.discovery.Validator;
import com.kontakt.sdk.android.ble.filter.eddystone.EddystoneFilter;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import java.util.List;

public class EddystoneDiscoverer extends AbstractDeviceDiscoverer<IEddystoneNamespace, IEddystoneDevice, EddystoneFilter> {

  private final Validator<IEddystoneDevice, IEddystoneNamespace> validator = new NamespaceValidator();
  private final EddystoneParser parser;

  public EddystoneDiscoverer(DiscoveryContract discoveryContract, ScanContext scanContext, FutureShufflesCache shufflesCache) {
    super(discoveryContract, scanContext, scanContext.getEddystoneNamespaces(), scanContext.getEddystoneFilters(), shufflesCache);
    this.parser = new EddystoneParser(scanContext);
  }

  @Override
  protected BluetoothDeviceEvent createEvent(EventType eventType, IEddystoneNamespace eddystoneNamespace, List<IEddystoneDevice> deviceList) {
    return new EddystoneDeviceEvent(eventType, eddystoneNamespace, deviceList);
  }

  @Override
  protected void onBeforeDeviceLost(IEddystoneDevice device) {
    parser.clearRssiCalculation(device.getAddress().hashCode());
  }

  @Override
  public void performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanRecord) {
    String deviceAddress = bluetoothDevice.getAddress();

    if (parser.isEnabled() && !parser.isValidEddystoneFrame(scanRecord)) {
      return;
    }

    EddystoneFrameType frameType = EddystoneFrameType.fromScanRecord(scanRecord);
    if (frameType == null) {
      return;
    }

    boolean frameParsed = parser.parseFrame(frameType, deviceAddress, scanRecord);
    if (!frameParsed) {
      return;
    }

    notifyDevicePresent(deviceAddress.hashCode(), System.currentTimeMillis());

    if (!parser.areTriggerFramesParsed(deviceAddress)) {
      return;
    }

    IEddystoneDevice device = parser.getEddystoneDevice(bluetoothDevice, rssi);
    if (device.isShuffled()) {
      resolveShuffled(device);
    } else {
      onShuffleResolved(device);
    }
  }

  @Override
  protected void onShuffleResolved(IEddystoneDevice device) {
    IEddystoneNamespace namespace = extractNamespace(device);
    if (namespace == null && getSpaceSet().contains(EddystoneNamespace.EVERYWHERE)) {
      namespace = EddystoneNamespace.EVERYWHERE;
    } else if (namespace == null) {
      return;
    }

    notifySpacePresent(namespace.hashCode(), System.currentTimeMillis());

    ReplacingArrayList<IEddystoneDevice> deviceList = getDevicesInSpace(namespace);
    if (deviceList == null) {
      deviceList = new ReplacingArrayList<>();
      insertDevicesIntoSpace(namespace, deviceList);
      onSpaceEnteredEvent(namespace);
    }

    if (!applyFilters(device)) {
      return;
    }

    if (deviceList.addOrReplace(device)) {
      onDeviceDiscoveredEvent(namespace, device);
    } else {
      onDevicesUpdatedEvent(namespace, deviceList);
    }
  }

  private IEddystoneNamespace extractNamespace(final IEddystoneDevice eddystone) {
    for (final IEddystoneNamespace namespace : getSpaceSet()) {
      if (validator.isValid(eddystone, namespace)) {
        return namespace;
      }
    }
    return null;
  }

  @Override
  public void disable() {
    parser.disable();
    super.disable();
  }
}
