package com.kontakt.sdk.android.ble.discovery;

import com.kontakt.sdk.android.ble.discovery.eddystone.EddystoneDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.ibeacon.IBeaconDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import com.kontakt.sdk.android.common.util.SecureProfileUtils;
import java.util.List;

/**
 * Utility methods for bluetooth devices discovery.
 */
public final class DiscoveryUtils {

  private DiscoveryUtils() {
  }

  /**
   * Calculates distance from remote device.
   *
   * @param txPower the tx power
   * @param rssi the rssi
   * @param profile device profile
   * @return the double
   * @see <a href="http://stackoverflow.com/questions/20416218/understanding-ibeacon-distancing" target="_blank">Stack Overflow answer</a>
   */
  public static double calculateDistance(int txPower, double rssi, DeviceProfile profile) {
    SDKPreconditions.checkNotNull(profile, "Profile can't be null");

    if (rssi == 0) {
      return -1.0;
    }

    int normalizedTxPower = (profile == DeviceProfile.EDDYSTONE ? txPower - 41 : txPower);
    double ratio = rssi * 1.0 / normalizedTxPower;
    if (ratio < 1.0) {
      return Math.pow(ratio, 10);
    } else {
      return (0.89976) * Math.pow(ratio, 7.7095) + 0.111;
    }
  }

  /**
   * Creates updated BluetoothDeviceEvent. Used internally.
   *
   * @param bluetoothDeviceEvent original event.
   * @param devices list of updated devices.
   * @return new BluetoothDeviceEvent with updated list of devices
   */
  static BluetoothDeviceEvent createUpdatedEvent(BluetoothDeviceEvent bluetoothDeviceEvent, List<? extends RemoteBluetoothDevice> devices) {
    switch (bluetoothDeviceEvent.getDeviceProfile()) {
      case IBEACON:
        IBeaconDeviceEvent iBeaconDeviceEvent = (IBeaconDeviceEvent) bluetoothDeviceEvent;
        List<IBeaconDevice> iBeaconDevices = (List<IBeaconDevice>) devices;
        return IBeaconDeviceEvent.of(iBeaconDeviceEvent, iBeaconDevices);
      case EDDYSTONE:
        EddystoneDeviceEvent eddystoneDeviceEvent = (EddystoneDeviceEvent) bluetoothDeviceEvent;
        List<IEddystoneDevice> eddystoneDevices = (List<IEddystoneDevice>) devices;
        return EddystoneDeviceEvent.of(eddystoneDeviceEvent, eddystoneDevices);
      case KONTAKT_SECURE:
        SecureProfileEvent secureProfileEvent = (SecureProfileEvent) bluetoothDeviceEvent;
        return SecureProfileEvent.of(secureProfileEvent, SecureProfileUtils.fromRemoteBluetoothDevices(devices));
      default:
        throw new IllegalArgumentException("Unsupported device profile!");
    }
  }
}
