package com.kontakt.sdk.android.ble.discovery;

import android.annotation.TargetApi;
import android.os.Build;
import android.util.SparseLongArray;
import com.android.internal.util.Predicate;
import com.kontakt.sdk.android.ble.cache.FutureShufflesCache;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.ble.util.SafeSparseLongArray;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Abstraction for Bluetooth device discoverers in the SDK. The abstraction provides
 * convenient interface whenever you need to implement your own Bluetooth device discoverer.
 *
 * @param <Space> Space type parameter which the discovered devices belong to
 * @param <Device> Bluetooth device type parameter
 */
@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public abstract class AbstractDeviceDiscoverer<Space, Device extends RemoteBluetoothDevice, Filter extends Predicate<Device>>
    implements BluetoothDeviceDiscoverer, ShuffledDevicesResolver.ResolveCallback {

  private final Map<Space, Long> lastCallbacksTimeMap = new HashMap<>();
  private final Map<Space, ReplacingArrayList<Device>> spaceDeviceListMap = new ConcurrentHashMap<>();
  private final SparseLongArray deviceTimestampArray = new SafeSparseLongArray();
  private final Collection<Filter> filters;
  private final SparseLongArray spaceTimestampArray;
  private final DiscoveryContract discoveryContract;
  private final ShuffledDevicesResolver shuffleResolver;
  private final Collection<Space> spaceSet;
  private final ScanContext scanContext;

  protected AbstractDeviceDiscoverer(DiscoveryContract discoveryContract, ScanContext scanContext, Collection<Space> spaceSet,
      Collection<Filter> filters, FutureShufflesCache shufflesCache) {
    this.discoveryContract = discoveryContract;
    this.scanContext = scanContext;
    this.spaceSet = spaceSet;
    this.filters = filters;
    this.spaceTimestampArray = new SafeSparseLongArray(spaceSet.size());
    this.shuffleResolver = new ShuffledDevicesResolver(this, shufflesCache);
    initCallbacksTimeMap(spaceSet);
  }

  private void initCallbacksTimeMap(Collection<Space> spaceSet) {
    for (Space space : spaceSet) {
      lastCallbacksTimeMap.put(space, 0L);
    }
  }

  protected abstract BluetoothDeviceEvent createEvent(EventType eventType, Space space, List<Device> deviceList);

  protected abstract void onBeforeDeviceLost(Device device);

  protected abstract void onShuffleResolved(Device device);

  protected void notifyDevicePresent(int deviceAddressHashcode, final long timestamp) {
    deviceTimestampArray.put(deviceAddressHashcode, timestamp);
  }

  protected void notifySpacePresent(final int spaceHashcode, final long timestamp) {
    spaceTimestampArray.put(spaceHashcode, timestamp);
  }

  protected Collection<Space> getSpaceSet() {
    return spaceSet;
  }

  protected ReplacingArrayList<Device> getDevicesInSpace(Space space) {
    return spaceDeviceListMap.get(space);
  }

  protected void resolveShuffled(RemoteBluetoothDevice device) {
    shuffleResolver.resolve(device);
  }

  @Override
  public void onResolved(RemoteBluetoothDevice device) {
    //noinspection unchecked
    onShuffleResolved((Device) device);
  }

  protected boolean applyFilters(Device device) {
    if (device == null) {
      return false;
    }

    if (filters.isEmpty()) {
      return true;
    }

    for (Filter filter : filters) {
      if (!filter.apply(device)) {
        return false;
      }
    }

    return true;
  }

  protected void insertDevicesIntoSpace(Space space, ReplacingArrayList<Device> deviceList) {
    spaceDeviceListMap.put(space, deviceList);
  }

  protected void onSpaceEnteredEvent(Space space) {
    discoveryContract.onEvent(createEvent(EventType.SPACE_ENTERED, space, Collections.<Device>emptyList()));
  }

  protected void onDeviceDiscoveredEvent(Space space, Device device) {
    discoveryContract.onEvent(createEvent(EventType.DEVICE_DISCOVERED, space, Collections.singletonList(device)));
  }

  protected void onDevicesUpdatedEvent(Space space, Collection<Device> deviceList) {
    final long devicesUpdateCallbackInterval = scanContext.getDeviceUpdateCallbackInterval();
    final long currentTimeMillis = System.currentTimeMillis();
    if (lastCallbacksTimeMap.get(space) + devicesUpdateCallbackInterval > currentTimeMillis) {
      return;
    }
    lastCallbacksTimeMap.put(space, currentTimeMillis);
    discoveryContract.onEvent(createEvent(EventType.DEVICES_UPDATE, space, new ArrayList<>(deviceList)));
  }

  void onDeviceLostEvent(Space space, Device device) {
    discoveryContract.onEvent(createEvent(EventType.DEVICE_LOST, space, Collections.singletonList(device)));
    shuffleResolver.onDeviceLost(device);
  }

  void onSpaceAbandonedEvent(Space space) {
    discoveryContract.onEvent(createEvent(EventType.SPACE_ABANDONED, space, Collections.<Device>emptyList()));
  }

  @Override
  public void evictInactiveDevices(long currentTimeMillis) {
    final long inactivityTimeout = scanContext.getActivityCheckConfiguration().getInactivityTimeout();
    for (Map.Entry<Space, ReplacingArrayList<Device>> entry : spaceDeviceListMap.entrySet()) {

      final Space space = entry.getKey();
      Iterator<Device> iterator = entry.getValue().iterator();
      while (iterator.hasNext()) {
        Device remoteDevice = iterator.next();
        long deviceTimestamp = deviceTimestampArray.get(remoteDevice.getAddress().hashCode(), -1L);
        if (deviceTimestamp != -1L) {
          long threshold = currentTimeMillis - deviceTimestamp;
          if (threshold > inactivityTimeout) {
            iterator.remove();
            onBeforeDeviceLost(remoteDevice);
            onDeviceLostEvent(space, remoteDevice);
            removeDeviceTimestamp(remoteDevice);
          }
        }
      }
    }

    evictInactiveRegions(inactivityTimeout, currentTimeMillis);
  }

  private void evictInactiveRegions(long inactivityTimeout, long currentTimeMillis) {
    final Set<Space> spaces = new HashSet<>(spaceDeviceListMap.keySet());

    for (final Space space : spaces) {
      final long regionTimestamp = spaceTimestampArray.get(space.hashCode(), -1L);
      if (regionTimestamp != -1L) {
        final long threshold = currentTimeMillis - regionTimestamp;
        if (threshold > inactivityTimeout) {
          spaceDeviceListMap.remove(space);
          onSpaceAbandonedEvent(space);
          removeSpaceTimestamp(space);
        }
      }
    }
  }

  private void removeDeviceTimestamp(Device remoteDevice) {
    int index = deviceTimestampArray.indexOfKey(remoteDevice.getAddress().hashCode());
    if (index >= 0) {
      deviceTimestampArray.removeAt(index);
    }
  }

  private void removeSpaceTimestamp(Space space) {
    int index = spaceTimestampArray.indexOfKey(space.hashCode());
    if (index >= 0) {
      spaceTimestampArray.removeAt(index);
    }
  }

  @Override
  public void disable() {
    shuffleResolver.disable();
  }
}
