package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.ble.connection.KontaktDeviceServiceStore;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.EddystoneUtils;
import java.util.UUID;

/**
 * Representation of all beacon characteristics discoverd during scan
 */
public class KontaktDeviceCharacteristics implements RemoteBluetoothDevice.Characteristics {

  private final UUID proximityUUID;
  private final int major;
  private final int minor;
  private final int powerLevel;
  private final long advertisingInterval;
  private final String batteryLevel;
  private final String manufacturerName;
  private final String modelName;
  private final String firmwareRevision;
  private final String hardwareRevision;
  private final DeviceProfile activeProfile;

  private final String namespaceId;
  private final String instanceId;
  private final String url;

  private final boolean secure;

  /**
   * The CREATOR Parcelable constant.
   */
  public static final Creator<KontaktDeviceCharacteristics> CREATOR = new Creator<KontaktDeviceCharacteristics>() {
    @Override
    public KontaktDeviceCharacteristics createFromParcel(Parcel source) {
      return new KontaktDeviceCharacteristics(source);
    }

    @Override
    public KontaktDeviceCharacteristics[] newArray(int size) {
      return new KontaktDeviceCharacteristics[size];
    }
  };

  /**
   * Instantiates a new Characteristics.
   *
   * @param serviceStore the service store
   */
  public KontaktDeviceCharacteristics(final KontaktDeviceServiceStore serviceStore) {

    String modelNameValue = null;

    try {
      modelNameValue = serviceStore.getPropagatedDeviceNameCharacteristic().getStringValue();
    } catch (Exception ignored) {
    }

    String manufacturerNameValue = null;

    try {
      manufacturerNameValue = serviceStore.getManufacturerNameCharacteristic().getStringValue();
    } catch (Exception ignored) {
    }

    String firmwareRevisionValue = null;

    try {
      firmwareRevisionValue = serviceStore.getFirmwareRevisionCharacteristic().getStringValue();
    } catch (Exception ignored) {
    }

    String hardwareRevisionValue = null;

    try {
      hardwareRevisionValue = serviceStore.getHardwareRevisionCharacteristic().getStringValue();
    } catch (Exception e) {
    }

    int powerLevelValue = -1;

    try {
      powerLevelValue = ConversionUtils.toPowerLevel(serviceStore.getPowerLevelCharacteristic().getValue());
    } catch (Exception e) {
    }

    String batteryLevelValue = null;

    try {
      batteryLevelValue = String.valueOf(serviceStore.getBatteryLevelCharacteristic().getIntValue()) + "%";
    } catch (Exception ignored) {
    }

    UUID proximityUUIDValue = null;

    try {
      proximityUUIDValue = serviceStore.getProximityCharacteristic().getUUIDValue();
    } catch (Exception ignored) {
    }

    int majorValue = -1;

    try {
      majorValue = ConversionUtils.asInt(serviceStore.getMajorCharacteristic().getValue());
    } catch (Exception ignored) {
    }

    int minorValue = -1;

    try {
      minorValue = ConversionUtils.asInt(serviceStore.getMinorCharacteristic().getValue());
    } catch (Exception ignored) {
    }

    long advertisingIntervalValue = -1;

    try {
      advertisingIntervalValue = Math.round(0.625 * serviceStore.getAdvertisingIntervalCharacteristic().getLongValue());
    } catch (Exception ignored) {
    }

    DeviceProfile deviceProfile = null;

    try {
      int activeProfileValue = ConversionUtils.asInt(serviceStore.getActiveProfileCharacteristic().getValue());
      deviceProfile = DeviceProfile.getActiveProfile(activeProfileValue);
    } catch (Exception ignored) {
    }

    String namespaceId = null;

    try {
      byte[] value = serviceStore.getNamespaceIdCharacteristic().getValue();
      namespaceId = EddystoneUtils.toStringFromHex(value);
    } catch (Exception ignored) {
    }

    String instanceId = null;

    try {
      byte[] instanceBytes = serviceStore.getInstanceIdCharacteristic().getValue();
      instanceId = EddystoneUtils.toStringFromHex(instanceBytes);
    } catch (Exception ignored) {
    }

    String url = null;

    try {
      byte[] urlBytes = serviceStore.getUrlCharacteristic().getValue();
      url = EddystoneUtils.deserializeUrl(urlBytes);
    } catch (Exception ignored) {
    }

    this.modelName = modelNameValue;
    this.manufacturerName = manufacturerNameValue;
    this.firmwareRevision = firmwareRevisionValue;
    this.hardwareRevision = hardwareRevisionValue;
    this.powerLevel = powerLevelValue;
    this.batteryLevel = batteryLevelValue;
    this.activeProfile = deviceProfile;
    this.advertisingInterval = advertisingIntervalValue;

    this.proximityUUID = proximityUUIDValue;
    this.major = majorValue;
    this.minor = minorValue;

    this.namespaceId = namespaceId;
    this.instanceId = instanceId;
    this.url = url;
    this.secure = serviceStore.getStoreType() == KontaktDeviceServiceStore.StoreType.SECURE;
  }

  KontaktDeviceCharacteristics(final Parcel source) {
    final Bundle bundle = source.readBundle();
    modelName = bundle.getString(Constants.Devices.NAME);
    manufacturerName = bundle.getString(Constants.Devices.MANUFACTURER_NAME);
    firmwareRevision = bundle.getString(Constants.Devices.FIRMWARE_REVISION);
    hardwareRevision = bundle.getString(Constants.Devices.HARDWARE_REVISION);
    powerLevel = bundle.getInt(Constants.Devices.TX_POWER);
    batteryLevel = bundle.getString(Constants.Devices.BATTERY);
    advertisingInterval = bundle.getLong(Constants.Devices.INTERVAL);
    activeProfile = (DeviceProfile) bundle.getSerializable(Constants.Devices.ACTIVE_PROFILE);

    proximityUUID = (UUID) bundle.getSerializable(Constants.Devices.PROXIMITY);
    major = bundle.getInt(Constants.Devices.MAJOR);
    minor = bundle.getInt(Constants.Devices.MINOR);

    namespaceId = bundle.getString(Constants.Eddystone.NAMESPACE);
    instanceId = bundle.getString(Constants.Eddystone.INSTANCE_ID);
    url = bundle.getString(Constants.Eddystone.URL);
    secure = bundle.getBoolean(Constants.Devices.SECURE);
  }

  @Override
  public UUID getProximityUUID() {
    return proximityUUID;
  }

  @Override
  public DeviceProfile getActiveProfile() {
    return activeProfile;
  }

  @Override
  public int getMajor() {
    return major;
  }

  @Override
  public int getMinor() {
    return minor;
  }

  @Override
  public int getPowerLevel() {
    return powerLevel;
  }

  @Override
  public long getAdvertisingInterval() {
    return advertisingInterval;
  }

  @Override
  public String getBatteryLevel() {
    return batteryLevel;
  }

  @Override
  public String getManufacturerName() {
    return manufacturerName;
  }

  @Override
  public String getModelName() {
    return modelName;
  }

  @Override
  public String getFirmwareRevision() {
    return firmwareRevision;
  }

  @Override
  public String getHardwareRevision() {
    return hardwareRevision;
  }

  @Override
  public String getUrl() {
    return url;
  }

  @Override
  public String getNamespaceId() {
    return namespaceId;
  }

  @Override
  public String getInstanceId() {
    return instanceId;
  }

  @Override
  public boolean isSecure() {
    return secure;
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    final Bundle bundle = new Bundle();
    bundle.putSerializable(Constants.Devices.PROXIMITY, proximityUUID);
    bundle.putString(Constants.Devices.NAME, modelName);
    bundle.putString(Constants.Devices.MANUFACTURER_NAME, manufacturerName);
    bundle.putString(Constants.Devices.FIRMWARE_REVISION, firmwareRevision);
    bundle.putString(Constants.Devices.HARDWARE_REVISION, hardwareRevision);
    bundle.putInt(Constants.Devices.TX_POWER, powerLevel);
    bundle.putString(Constants.Devices.BATTERY, batteryLevel);
    bundle.putInt(Constants.Devices.MAJOR, major);
    bundle.putInt(Constants.Devices.MINOR, minor);
    bundle.putLong(Constants.Devices.INTERVAL, advertisingInterval);
    bundle.putSerializable(Constants.Devices.ACTIVE_PROFILE, activeProfile);
    bundle.putString(Constants.Eddystone.NAMESPACE, namespaceId);
    bundle.putString(Constants.Eddystone.INSTANCE_ID, instanceId);
    bundle.putString(Constants.Eddystone.URL, url);
    bundle.putBoolean(Constants.Devices.SECURE, secure);
    dest.writeBundle(bundle);
  }
}
