package com.kontakt.sdk.android.ble.device;

import android.os.Bundle;
import android.os.Parcel;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.Constants;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkArgument;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Object representing Eddystone Namespace
 * Namespace or Secure Namespace is required for setting Eddystone Namespace.
 *
 * <p>
 * Example usage:
 * <pre>
 * EddystoneNamespace namespace = new EddystoneNamespace.Builder()
 *  .identifier("The Office")
 *  .namespace("f7826da6bc5b71e0893e")
 *  .build();
 * </pre>
 */
public class EddystoneNamespace implements IEddystoneNamespace {

  /**
   * Value marking that any instance ID is acceptable.
   */
  public static final String ANY_INSTANCE_ID = "Any instance ID";

  /**
   * Predefined constant specifying that every visible remote Eddystone device should be notified
   * by Android device regardless of its belonging to the specific namespace.
   */
  public static EddystoneNamespace EVERYWHERE = new Builder()
      .identifier("Everywhere")
      .namespace(KontaktSDK.DEFAULT_KONTAKT_NAMESPACE_ID)
      .secureNamespace(null)
      .instanceId(ANY_INSTANCE_ID)
      .build();

  public static final Creator<EddystoneNamespace> CREATOR = new Creator<EddystoneNamespace>() {
    @Override
    public EddystoneNamespace createFromParcel(Parcel source) {
      Bundle bundle = source.readBundle(getClass().getClassLoader());
      return new Builder().identifier(bundle.getString(Constants.Eddystone.IDENTIFIER))
          .namespace(bundle.getString(Constants.Eddystone.NAMESPACE))
          .secureNamespace(bundle.getString(Constants.Eddystone.SECURE_NAMESPACE))
          .instanceId(bundle.getString(Constants.Eddystone.INSTANCE_ID))
          .build();
    }

    @Override
    public EddystoneNamespace[] newArray(int size) {
      return new EddystoneNamespace[size];
    }
  };

  private final String identifier;
  private final String namespace;
  private final String secureNamespace;
  private final String instanceId;

  public static Builder builder() {
    return new Builder();
  }

  EddystoneNamespace(Builder builder) {
    checkNotNull(builder.identifier, "Identifier cannot be null");
    checkArgument(builder.namespace != null || builder.secureNamespace != null, "You must set either Namespace or SecureNamespace.");
    this.identifier = builder.identifier;
    this.namespace = builder.namespace;
    this.secureNamespace = builder.secureNamespace;
    this.instanceId = builder.instanceId;
  }

  @Override
  public String getIdentifier() {
    return identifier;
  }

  @Override
  public String getNamespace() {
    return namespace;
  }

  @Override
  public String getSecureNamespace() {
    return secureNamespace;
  }

  @Override
  public String getInstanceId() {
    return instanceId;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    EddystoneNamespace that = (EddystoneNamespace) o;

    if (!identifier.equals(that.identifier)) return false;
    if (namespace != null ? !namespace.equals(that.namespace) : that.namespace != null) return false;
    if (secureNamespace != null ? !secureNamespace.equals(that.secureNamespace) : that.secureNamespace != null) return false;
    return instanceId != null ? instanceId.equals(that.instanceId) : that.instanceId == null;
  }

  @Override
  public int hashCode() {
    int result = identifier.hashCode();
    result = 31 * result + (namespace != null ? namespace.hashCode() : 0);
    result = 31 * result + (secureNamespace != null ? secureNamespace.hashCode() : 0);
    result = 31 * result + (instanceId != null ? instanceId.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    StringBuilder namespaceDesc = new StringBuilder();
    namespaceDesc.append("[");
    if (namespace != null) {
      namespaceDesc.append(namespace);
    } else if (secureNamespace != null) {
      namespaceDesc.append(secureNamespace);
    }
    namespaceDesc.append(", ");

    if (ANY_INSTANCE_ID.equals(instanceId)) {
      namespaceDesc.append("ANY_INSTANCE_ID");
    } else {
      namespaceDesc.append(instanceId);
    }
    namespaceDesc.append("]");

    return namespaceDesc.toString();
  }

  @Override
  public int compareTo(IEddystoneNamespace another) {
    if (this == another) {
      return 0;
    }
    return identifier.compareTo(another.getIdentifier());
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel parcel, int flags) {
    Bundle bundle = new Bundle(getClass().getClassLoader());
    bundle.putString(Constants.Eddystone.IDENTIFIER, identifier);
    bundle.putString(Constants.Eddystone.NAMESPACE, namespace);
    bundle.putString(Constants.Eddystone.SECURE_NAMESPACE, secureNamespace);
    bundle.putString(Constants.Eddystone.INSTANCE_ID, instanceId);

    parcel.writeBundle(bundle);
  }

  public static class Builder {

    String identifier;
    String namespace;
    String secureNamespace;
    String instanceId = ANY_INSTANCE_ID;

    public Builder identifier(String identifier) {
      this.identifier = identifier;
      return this;
    }

    public Builder namespace(String namespace) {
      this.namespace = namespace;
      return this;
    }

    public Builder secureNamespace(String secureNamespace) {
      this.secureNamespace = secureNamespace;
      return this;
    }

    public Builder instanceId(String instanceId) {
      this.instanceId = instanceId;
      return this;
    }

    public EddystoneNamespace build() {
      return new EddystoneNamespace(this);
    }
  }
}
