package com.kontakt.sdk.android.ble.connection;

import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import java.util.UUID;

/**
 * Syncable Kontakt device connection combines {@link KontaktDeviceConnection} and {@link KontaktCloud} functionality.
 * The syncable iBeacon connection allows for immediate pushes to both Bluetooth iBeacon and its
 * representation in kontakt.io Web API.
 */
public interface SyncableKontaktDeviceConnection {

  /**
   * Connects to beacon device
   *
   * @return true, if the connection attempt was initiated succesfully
   */
  boolean connectToDevice();

  /**
   * Returns true if Connection with Beacon is established.
   *
   * @return the boolean flag indicating whether connection is established.
   */
  boolean isConnectedToDevice();

  /**
   * Closes Beacon connection releases provided resources.
   */
  void close();

  /**
   * Gets devices that the connection is established with.
   *
   * @return the iBeacon device.
   */
  RemoteBluetoothDevice getDevice();

  /**
   * Returns true if Beacon was successfully authenticated.
   *
   * @return the boolean flag indicating whether beacon was authenticated.
   */
  boolean isDeviceAuthenticated();

  /**
   * Writes minor value and tries to sync with REST API
   *
   * @param value the minor value to set
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteMinor(int value, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes major value and tries to sync with REST API
   *
   * @param value the major value
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteMajor(int value, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes power level and tries to sync with REST API
   *
   * @param value the power level
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwritePowerLevel(int value, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes proximity uuid value and tries to sync with REST API
   *
   * @param proximityUUID the proximity {@link UUID}
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteProximityUUID(UUID proximityUUID, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes password and tries to sync with REST API
   *
   * @param password the password for device
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwritePassword(String password, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes advertising interval and tries to sync with REST API
   *
   * @param interval advertising interval
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteAdvertisingInterval(long interval, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes model name and tries to sync with REST API
   *
   * @param modelName model name
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteModelName(String modelName, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Resets device so that connection is established once again.
   * <br>
   * For more information see kontakt.io Datasheet - version 2.0.
   * <br>
   * The result callback is invoked by WriteListener.
   *
   * @param writeListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
   */
  void resetDevice(WriteListener writeListener);

  /**
   * Overwrites url
   *
   * @param url the url
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteUrl(String url, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Overwrites namespae
   *
   * @param namespaceId the namespace
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteNamespace(String namespaceId, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Overwrites instanceId
   *
   * @param instanceId the instanceId
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void overwriteInstanceId(String instanceId, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Writes secure config and tries to sync with REST API
   *
   * @param secureConfig the secureConfig
   * @param syncWriteListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteListener}
   */
  void applySecureConfig(Config secureConfig, SyncableKontaktDeviceConnectionImpl.SyncWriteListener syncWriteListener);

  /**
   * Applies pending configuration ({@link Config}) for Beacon.
   * <br>
   * * It is possible to encounter problems during batch (the Config in this case)
   * write operation which results in launching error handle callback method.
   * Once the write batch operation failes, the Write Batch Processor
   * will attempt performing rollback write batch operation aiming at
   * restoring original Beacon configuration.
   * The rollback operation however may not succeed as well leaving Beacon device
   * changed partially.
   * Therefore it is strongly advisable to write batches in relatively close distance
   * to Beacon device. Thus, the risk of leaving Beacon in unspecified state may
   * be avoided.
   * <br>
   * If batch write succeed attempt to sync with REST API will be done
   *
   * @param config {@link Config}
   * @param syncWriteBatchListener {@link SyncableKontaktDeviceConnectionImpl.SyncWriteBatchListener}
   */
  void applyConfig(Config config, SyncableKontaktDeviceConnectionImpl.SyncWriteBatchListener<Config> syncWriteBatchListener);
}
