package com.kontakt.sdk.android.ble.connection;

import android.util.Base64;
import com.kontakt.sdk.android.ble.security.EncryptedWriteRequest;
import com.kontakt.sdk.android.ble.security.ResponseCode;
import com.kontakt.sdk.android.ble.security.WriteRequest;
import com.kontakt.sdk.android.ble.security.auth.AuthToken;
import com.kontakt.sdk.android.ble.security.exception.InvalidConfigException;
import com.kontakt.sdk.android.ble.security.parser.SimpleResponseParser;
import com.kontakt.sdk.android.common.model.Config;

import static com.kontakt.sdk.android.ble.connection.SecureConfigMapper.toSecureProperties;

/**
 * Class is taking care of applying offline secure config.
 * Operation should consist of reading beacon parameters, validating config and finally aplying new configuration if everything looks ok.
 */
class OfflineSecurityService {

  KontaktDeviceConnection connection;
  WriteListener writeListener = WriteListener.NOOP_LISTENER;
  AuthToken authToken;
  Config configToApply;

  OfflineSecurityService(KontaktDeviceConnection kontaktDeviceConnection) {
    this.connection = kontaktDeviceConnection;
  }

  void close() {
    connection = null;
  }

  void applySecureConfig(Config config, AuthToken token, WriteListener writeListener) {
    this.configToApply = config;
    this.writeListener = writeListener;
    this.authToken = token;
    connection.readAll(authToken, internalReadListener);
  }

  void validateConfig(Config readConfig) {
    //Offline security does not allow turning shuffling on
    if (configToApply.isShuffled()) {
      writeListener.onWriteFailure(ErrorCause.OPERATION_NOT_ALLOWED);
      return;
    }
    //If the beacon is shuffled some properties should not be changed to avoid cloud synchronization problems.
    if (readConfig.isShuffled() && isAnyForbiddenPropertyChanged(configToApply)) {
      writeListener.onWriteFailure(ErrorCause.OPERATION_NOT_ALLOWED);
      return;
    }

    sendSecureConfig(configToApply);
  }

  void sendSecureConfig(Config config) {
    int timestampToken = (int) (System.currentTimeMillis() / 1000L);
    WriteRequest request = new EncryptedWriteRequest(timestampToken, toSecureProperties(config), authToken.getPassword());
    connection.applySecureConfig(request.getBase64Data(), internalWriteListener);
  }

  void checkWriteResponse(WriteListener.WriteResponse response) {
    byte[] decodedBase = Base64.decode(response.getExtra(), Base64.DEFAULT);
    try {
      SimpleResponseParser responseParser = SimpleResponseParser.of(decodedBase, authToken.getPassword());
      ResponseCode result = responseParser.getResult();
      switch (result) {
        case OK:
          writeListener.onWriteSuccess(response);
          break;
        case VERIFICATION_FAILED:
          writeListener.onWriteFailure(ErrorCause.VERIFICATION_FAILED);
          break;
        case AUTHORIZATION_FAILED:
          writeListener.onWriteFailure(ErrorCause.AUTHORIZATION_FAILED);
          break;
        case ID_NOT_FOUND:
          writeListener.onWriteFailure(ErrorCause.ID_NOT_FOUND);
          break;
        default:
          writeListener.onWriteFailure(ErrorCause.INCORRECT_RESPONSE);
      }
    } catch (InvalidConfigException e) {
      writeListener.onWriteFailure(ErrorCause.INCORRECT_RESPONSE);
    }
  }

  private boolean isAnyForbiddenPropertyChanged(Config configToApply) {
    return configToApply.getProximity() != null
        || configToApply.getMajor() > 0
        || configToApply.getMinor() > 0
        || configToApply.getNamespace() != null
        || configToApply.getInstanceId() != null
        || configToApply.isShuffled();
  }

  void unregisterListeners() {
    if (connection != null) {
      ((KontaktDeviceConnectionImpl) connection).unregisterAllListeners();
    }
  }

  private final WriteListener internalWriteListener = new WriteListener() {
    @Override
    public void onWriteSuccess(WriteResponse response) {
      if (connection != null) {
        checkWriteResponse(response);
      }
    }

    @Override
    public void onWriteFailure(ErrorCause cause) {
      writeListener.onWriteFailure(cause);
    }
  };

  private final ReadListener<Config> internalReadListener = new ReadListener<Config>() {
    @Override
    public void onReadSuccess(Config readConfig) {
      unregisterListeners();
      validateConfig(readConfig);
    }

    @Override
    public void onReadFailure(ErrorCause cause) {
      unregisterListeners();
      writeListener.onWriteFailure(cause);
    }
  };
}
