package com.kontakt.sdk.android.ble.cache;

import com.kontakt.sdk.android.cloud.KontaktCloud;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.common.model.Device;
import com.kontakt.sdk.android.common.model.EddystoneFutureUID;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

class EddystoneApiResolveStrategy implements EddystoneResolveStrategy {

  private final KontaktCloud kontaktCloud;

  EddystoneApiResolveStrategy(KontaktCloud kontaktCloud) {
    this.kontaktCloud = kontaktCloud;
  }

  @Override
  public List<EddystoneFutureUID> resolve(final Map<EddystoneUid, EddystoneResolveRequest> requests) throws Exception {
    final List<EddystoneUid> toResolve = eddystonesToResolve(requests);
    if (toResolve.isEmpty()) {
      return Collections.emptyList();
    }

    final List<EddystoneFutureUID> futureIds = resolveEddystones(toResolve);
    updateRequests(requests, futureIds);
    return futureIds;
  }

  private List<EddystoneUid> eddystonesToResolve(final Map<EddystoneUid, EddystoneResolveRequest> requests) {
    final List<EddystoneUid> toResolve = new ArrayList<>();
    for (Map.Entry<EddystoneUid, EddystoneResolveRequest> entry : requests.entrySet()) {
      final EddystoneResolveRequest request = entry.getValue();
      if (ResolveRequestStatus.RESOLVED == request.getStatus()) {
        continue;
      }
      toResolve.add(entry.getKey());
    }
    return toResolve;
  }

  private List<EddystoneFutureUID> resolveEddystones(final List<EddystoneUid> uids) throws IOException, KontaktCloudException {
    final Devices cloudResult = kontaktCloud.devices().eddystones().withIds(uids).execute();
    final List<Device> devices = cloudResult.getContent();

    final List<EddystoneFutureUID> futureIds = new ArrayList<>();
    for (Device device : devices) {
      EddystoneFutureUID eddystoneFutureUID = device.extractEddystoneFutureUID();
      if (eddystoneFutureUID != null) {
        futureIds.add(eddystoneFutureUID);
      }
    }
    return futureIds;
  }

  private void updateRequests(Map<EddystoneUid, EddystoneResolveRequest> requests, List<EddystoneFutureUID> futureIds) {
    for (EddystoneFutureUID id : futureIds) {
      updateRequest(requests, id);
    }
  }

  private void updateRequest(Map<EddystoneUid, EddystoneResolveRequest> requests, EddystoneFutureUID id) {
    EddystoneResolveRequest request = requests.get(id.getQueriedBy());
    request.setResolvedBy(ResolverType.API);
    if (ResolveRequestStatus.IGNORED == request.getStatus()) {
      return;
    }
    request.setStatus(ResolveRequestStatus.RESOLVED);
  }
}
