package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

/**
 * Model class that represents JSON structure of Kontakt.io trigger's context.
 * To create new instance of this class, please use {@link Trigger.Builder}.
 */
public class TriggerContext implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<TriggerContext> CREATOR = new Creator<TriggerContext>() {
    @Override
    public TriggerContext createFromParcel(Parcel source) {
      return new TriggerContext(source);
    }

    @Override
    public TriggerContext[] newArray(int size) {
      return new TriggerContext[size];
    }
  };

  private final String cooldown;
  private final String sourceId;
  private final Proximity proximity;
  private final String trackingId;

  public static Builder builder() {
    return new Builder();
  }

  TriggerContext(Builder builder) {
    this.cooldown = builder.cooldown;
    this.sourceId = builder.sourceId;
    this.proximity = builder.proximity;
    this.trackingId = builder.trackingId;
  }

  protected TriggerContext(Parcel in) {
    this.cooldown = in.readString();
    this.sourceId = in.readString();
    this.proximity = (Proximity) in.readSerializable();
    this.trackingId = in.readString();
  }

  private TriggerContext() {
    this(new Builder());
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof TriggerContext)) {
      return false;
    }
    final TriggerContext triggerContext = (TriggerContext) object;

    return SDKEqualsBuilder.start()
        .equals(cooldown, triggerContext.cooldown)
        .equals(sourceId, triggerContext.sourceId)
        .equals(proximity, triggerContext.proximity)
        .equals(trackingId, triggerContext.trackingId)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(cooldown).append(sourceId).append(proximity).append(trackingId).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.cooldown);
    dest.writeString(this.sourceId);
    dest.writeSerializable(this.proximity);
    dest.writeString(this.trackingId);
  }

  public String getCooldown() {
    return cooldown;
  }

  public String getSourceId() {
    return sourceId;
  }

  public Proximity getProximity() {
    return proximity;
  }

  public String getTrackingId() {
    return trackingId;
  }

  /**
   * Builder class that is used to build {@link TriggerContext} instances from values configured by the setters.
   */
  public static class Builder {

    String cooldown;
    String sourceId;
    Proximity proximity;
    String trackingId;

    public Builder cooldown(final String cooldown) {
      this.cooldown = cooldown;
      return this;
    }

    public Builder sourceId(final String sourceId) {
      this.sourceId = sourceId;
      return this;
    }

    public Builder proximity(final Proximity proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder trackingId(final String trackingId) {
      this.trackingId = trackingId;
      return this;
    }

    public TriggerContext build() {
      return new TriggerContext(this);
    }
  }
}
