package com.kontakt.sdk.android.common;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import static com.kontakt.sdk.android.common.util.ArrayUtils.concat;
import static java.util.Arrays.asList;

/**
 * Representation of available firmware revisions
 */
public final class FirmwareRevisions {

  /**
   * Set of Beacon Pro firmwares revisions
   */
  private static final String[] BEACON_PRO_VERSIONS = new String[] { "1.1", "1.2", "1.3", "1.4", "1.5", "1.6", "1.7", "1.8" };

  /**
   * Set of Beacon Pro firmwares revisions that support RTC power saving and reading light sensor
   */
  public static final Set<String> BEACON_PRO_RTC_LIGHTSENSOR_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "1.7", "1.8" })));

  /**
   * Set of firmware revisions of beacon devices with many writable characteristic available
   */
  public static final Set<String> NORMAL_MODE_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "2.5", "2.6", "2.7", "2.8", "2.9", "3.0", "3.1" })));

  /**
   * Set of firmware revisions of beacon devices with single writable characteristic available
   */
  public static final Set<String> SECURE_MODE_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(concat(new String[] { "4.0", "4.1" }, BEACON_PRO_VERSIONS))));

  /**
   * Set of firmware revisions of Beacon Pro devices.
   */
  public static final Set<String> BEACON_PRO_FIRMWARE_VERSIONS = Collections.unmodifiableSet(new HashSet<>(asList(BEACON_PRO_VERSIONS)));

  /**
   * Set of firmware revisions of beacon devices with only IBeacon profile available
   */
  public static final Set<String> NON_SWITCHABLE_PROFILE_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "2.5", "2.6", "2.7", "2.8", "2.9" })));

  /**
   * Set of firmware revisions of beacon devices available to switch between IBeacon and Eddystone profile
   */
  public static final Set<String> SWITCHABLE_PROFILE_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(concat(new String[] { "3.0", "3.1", "4.0", "4.1" }, BEACON_PRO_VERSIONS))));

  /**
   * Set of firmware revisions of beacon devices which support only iBeacon profile
   */
  public static final Set<String> IBEACON_PROFILE_FIRMWARE_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "2.5", "2.6", "2.7" })));

  /**
   * Set of firmware revisions of beacon devices which support only Eddystone profile
   */
  public static final Set<String> EDDYSTONE_PROFILE_FIRMWARE_VERSIONS = Collections.unmodifiableSet(new HashSet<>(asList(new String[] { "2.9" })));

  /**
   * Set of firmware revisions of beacon devices which support interleaving.
   */
  public static final Set<String> INTERLEAVING_PACKETS_VERSIONS =
      Collections.unmodifiableSet(new HashSet<>(asList(concat(new String[] { "4.1" }, BEACON_PRO_VERSIONS))));

  /**
   * Checks if provided firmware versions supports switching profiles
   *
   * @param firmwareVersion version of beacon device
   * @return true if supported or false if not supported
   */
  public static boolean isSwitchable(String firmwareVersion) {
    return !NON_SWITCHABLE_PROFILE_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware versions is applicable only for iBeacon profile
   *
   * @param firmwareVersion version od beacon device
   * @return true if iBeacon only, false if Eddystone only or switchable
   */
  public static boolean isIBeaconOnly(String firmwareVersion) {
    return IBEACON_PROFILE_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware versions is applicable only for Eddystone profile
   *
   * @param firmwareVersion version od beacon device
   * @return true if Eddystone only, false if iBeacon only or switchable
   */
  public static boolean isEddystoneOnly(String firmwareVersion) {
    return EDDYSTONE_PROFILE_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware version is eligible for multiple packets.
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isInterleavingAvailable(String firmwareVersion) {
    return INTERLEAVING_PACKETS_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware version is eligible for multiple packets.
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isSecure(String firmwareVersion) {
    return SECURE_MODE_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware version is a beacon pro version.
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isPro(String firmwareVersion) {
    return BEACON_PRO_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }

  /**
   * Checks if provided firmware version is one of the older firmwares (2.5 to 3.1)
   *
   * @param firmwareVersion firmware version od beacon device
   * @return true or false
   */
  public static boolean isNormal(String firmwareVersion) {
    return NORMAL_MODE_FIRMWARE_VERSIONS.contains(firmwareVersion);
  }
}
