package com.kontakt.sdk.android.cloud.api.executor.triggers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.TriggersApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.TriggersService;
import com.kontakt.sdk.android.cloud.response.paginated.Triggers;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Request executor provided by {@link TriggersApi}. Use this class if you want to fetch
 * triggers through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Triggers triggers = kontaktCloud.triggers().fetch()
 *      .startIndex(5)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class TriggersRequestExecutor extends PaginatedRequestExecutor<Triggers> {

  private final TriggersService triggersService;
  private final List<UUID> triggersIds = new ArrayList<>();

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param triggersService the triggers API service.
   */
  public TriggersRequestExecutor(final TriggersService triggersService) {
    this.triggersService = triggersService;
  }

  /**
   * Use that method if you want to fetch particular triggers by ID.
   *
   * @param ids triggers unique identifiers.
   * @return this request executor.
   */
  public TriggersRequestExecutor withIds(final UUID... ids) {
    return withIds(Arrays.asList(ids));
  }

  /**
   * Use that method if you want to fetch particular triggers by ID.
   *
   * @param ids triggers unique identifiers.
   * @return this request executor.
   */
  public TriggersRequestExecutor withIds(final List<UUID> ids) {
    checkNotNull(ids, "Ids are null");
    this.triggersIds.addAll(ids);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TriggersRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TriggersRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TriggersRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TriggersRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TriggersRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Triggers> prepareCall() {
    final Call<Triggers> call;
    if (eTag != null) {
      call = triggersService.getTriggers(params(), eTag);
    } else {
      call = triggersService.getTriggers(params());
    }
    return call;
  }

  @Override
  protected Map<String, String> params() {
    if (triggersIds.isEmpty()) {
      return super.params();
    }
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Triggers.TRIGGER_ID_PARAMETER, StringUtils.join(triggersIds, ","));
    params.putAll(super.params());
    return params;
  }
}
