package com.kontakt.sdk.android.cloud.api.executor.triggers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.TriggersApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.TriggersService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Trigger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link TriggersApi}. Use this class if you want to create
 * trigger through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Trigger trigger = new Trigger.Builder()
 *      .name("Entered office")
 *      .build();
 *   kontaktCloud.triggers().create(trigger).execute();
 *   </code>
 * </pre>
 */
public class CreateTriggerRequestExecutor extends RequestExecutor<Trigger> {

  private final TriggersService triggersService;
  private final Trigger trigger;

  /**
   * Constructs request executor initialized with corresponding service class and trigger object.
   *
   * @param triggersService the triggers API service.
   * @param trigger the trigger entity.
   */
  public CreateTriggerRequestExecutor(final TriggersService triggersService, final Trigger trigger) {
    this.triggersService = triggersService;
    this.trigger = trigger;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Trigger execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Trigger> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Trigger> prepareCall() {
    return triggersService.create(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(trigger.getName() != null, "Cannot create trigger - specify name");
    SDKPreconditions.checkState(trigger.getType() != null, "Cannot create trigger - specify type");
    SDKPreconditions.checkState(trigger.getContext() != null, "Cannot create trigger - specify context");
    SDKPreconditions.checkState(trigger.getContext().getTrackingId() != null, "Cannot create trigger - specify tracking ID");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Triggers.NAME_PARAMETER, trigger.getName());
    params.put(CloudConstants.Triggers.TYPE_PARAMETER, trigger.getType().name());
    params.put(CloudConstants.Triggers.CONTEXT_TRACKING_ID_PARAMETER, trigger.getContext().getTrackingId());
    if (trigger.getExecutor() != null) {
      params.put(CloudConstants.Triggers.EXECUTOR_PARAMETER, trigger.getExecutor().name());
    }
    if (trigger.getContext().getProximity() != null) {
      params.put(CloudConstants.Triggers.CONTEXT_PROXIMITY_PARAMETER, trigger.getContext().getProximity().name());
    }
    if (trigger.getContext().getSourceId() != null) {
      params.put(CloudConstants.Triggers.CONTEXT_SOURCE_ID_PARAMETER, trigger.getContext().getSourceId());
    }
    return params;
  }
}
