package com.kontakt.sdk.android.cloud.api.executor.managers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ManagersApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ManagersService;
import com.kontakt.sdk.android.common.model.Manager;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link ManagersApi}. Use this class if you want to update
 * managers through fluent API in chained fashion, for instance:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Manager updatedManager = new Manager.Builder()
 *      .email("admin.kontakter@kontakt.io")
 *      .role(Manager.Role.ADMIN)
 *      .build();
 *   kontaktCloud.managers().update(ID)
 *      .with(updatedManager)
 *      .execute();
 *   </code>
 * </pre>
 */
public class UpdateManagerRequestExecutor extends RequestExecutor<String> {

  private final ManagersService managersService;

  private final UUID managerId;
  private Manager manager;

  /**
   * Constructs request executor initialized with corresponding service class and manager unique ID.
   *
   * @param managersService the managers API service.
   * @param managerId manager unique identifier.
   */
  public UpdateManagerRequestExecutor(final ManagersService managersService, final UUID managerId) {
    this.managersService = managersService;
    this.managerId = managerId;
  }

  /**
   * Specifies manager data to update.
   *
   * @param manager updated manager data.
   * @return this request executor.
   */
  public UpdateManagerRequestExecutor with(final Manager manager) {
    SDKPreconditions.checkNotNull(manager, "manager cannot be null");
    this.manager = manager;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return managersService.updateManager(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Managers.MANAGER_ID_PARAMETER, managerId.toString());
    if (manager.getFirstName() != null) {
      params.put(CloudConstants.Managers.FIRST_NAME_PARAMETER, manager.getFirstName());
    }
    if (manager.getLastName() != null) {
      params.put(CloudConstants.Managers.LAST_NAME_PARAMETER, manager.getLastName());
    }
    if (manager.getEmail() != null) {
      params.put(CloudConstants.Managers.EMAIL_PARAMETER, manager.getEmail());
    }
    if (manager.getRole() != null) {
      params.put(CloudConstants.Managers.ROLE_PARAMETER, manager.getRole().name());
    }

    return params;
  }
}
