package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to unassign
 * devices from venue through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   kontaktCloud.devices().unassignFromVenue(IDs).execute();
 *   </code>
 * </pre>
 */
public class UnassignDeviceRequestExecutor extends RequestExecutor<String> {

  private final DevicesService devicesService;

  private final String[] uniqueIds;

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   *
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public UnassignDeviceRequestExecutor(final DevicesService devicesService, final String... uniqueIds) {
    this.devicesService = devicesService;
    this.uniqueIds = uniqueIds;
  }

  /**
   * Constructs request executor initialized with corresponding service class and device unique IDs.
   *
   * @param devicesService the devices API service.
   * @param uniqueIds device unique identifiers.
   */
  public UnassignDeviceRequestExecutor(final DevicesService devicesService, final List<String> uniqueIds) {
    this.devicesService = devicesService;
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return devicesService.unassignDevice(params());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    return params;
  }
}
