package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.Access;
import com.kontakt.sdk.android.common.model.EddystoneUid;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.model.SecureProfileUid;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to get
 * Eddystone devices through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Devices eddystones = kontaktCloud.devices().eddystones()
 *      .withIds(eddystoneUIDs)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class EddystonesRequestExecutor extends DevicesBaseRequestExecutor {

  private final DevicesService devicesService;

  private String[] eddystoneUIDs;
  private String[] eddystoneEIDs;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param devicesService the devices API service.
   */
  public EddystonesRequestExecutor(final DevicesService devicesService) {
    this.devicesService = devicesService;
    this.profile = DeviceProfile.EDDYSTONE;
  }

  /**
   * Specifies Eddystone unique IDs. Use that method if you want to fetch particular Eddystone devices by ID.
   *
   * @param eddystoneUIDs Eddystone unique identifiers.
   * @return this request executor.
   */
  public EddystonesRequestExecutor withIds(final EddystoneUid... eddystoneUIDs) {
    SDKPreconditions.checkNotNull(eddystoneUIDs, "eddystone UIDs cannot be null");
    final int size = eddystoneUIDs.length;
    this.eddystoneUIDs = new String[size];
    for (int i = 0; i < size; i++) {
      this.eddystoneUIDs[i] = eddystoneUIDs[i].toString();
    }
    return this;
  }

  /**
   * Specifies Eddystone unique IDs. Use that method if you want to fetch particular Eddystone devices by ID.
   *
   * @param eddystoneUIDs Eddystone unique identifiers.
   * @return this request executor.
   */
  public EddystonesRequestExecutor withIds(final List<EddystoneUid> eddystoneUIDs) {
    SDKPreconditions.checkNotNull(eddystoneUIDs, "eddystone UIDs cannot be null");
    final int size = eddystoneUIDs.size();
    this.eddystoneUIDs = new String[size];
    for (int i = 0; i < size; i++) {
      EddystoneUid eddystoneUID = eddystoneUIDs.get(i);
      this.eddystoneUIDs[i] = eddystoneUID.toString();
    }
    return this;
  }

  /**
   * Specifies Eddystone EID frames. Use that method if you want to fetch particular Eddystone devices by EID frame.
   *
   * @param eddystoneEIDs Eddystone unique identifiers.
   * @return this request executor.
   */
  public EddystonesRequestExecutor withEids(final String... eddystoneEIDs) {
    SDKPreconditions.checkNotNull(eddystoneEIDs, "eddystone EIDs cannot be null");
    final int size = eddystoneEIDs.length;
    this.eddystoneEIDs = new String[size];
    for (int i = 0; i < size; i++) {
      this.eddystoneEIDs[i] = eddystoneEIDs[i];
    }
    return this;
  }

  /**
   * Specifies Eddystone EID frames. Use that method if you want to fetch particular Eddystone devices by EID frame.
   *
   * @param eddystoneEIDs Eddystone unique identifiers.
   * @return this request executor.
   */
  public EddystonesRequestExecutor withEids(final List<String> eddystoneEIDs) {
    SDKPreconditions.checkNotNull(eddystoneEIDs, "eddystone UIDs cannot be null");
    final int size = eddystoneEIDs.size();
    this.eddystoneEIDs = new String[size];
    for (int i = 0; i < size; i++) {
      String eid = eddystoneEIDs.get(i);
      this.eddystoneEIDs[i] = eid;
    }
    return this;
  }

  /**
   * Specifies SecureProfile shuffled unique IDs. Use that method if you want to fetch particular Devices by SecureProifle ID.
   *
   * @param secureProfileUids Eddystone unique identifiers.
   * @return this request executor.
   */
  public EddystonesRequestExecutor withSecureProfileIds(final List<SecureProfileUid> secureProfileUids) {
    SDKPreconditions.checkNotNull(secureProfileUids, "Secure profile UIDs cannot be null");
    final int size = secureProfileUids.size();
    this.eddystoneUIDs = new String[size];
    for (int i = 0; i < size; i++) {
      SecureProfileUid uid = secureProfileUids.get(i);
      this.eddystoneUIDs[i] = uid.toString();
    }
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Devices> prepareCall() {
    final Call<Devices> call;
    if (eTag != null) {
      call = devicesService.getDevices(params(), eTag);
    } else {
      call = devicesService.getDevices(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor startIndex(final int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor maxResult(final int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor filter(final String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor access(final Access access) {
    super.access(access);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor managerIds(final UUID... managerIds) {
    super.managerIds(managerIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor managerIds(final List<UUID> managerIds) {
    super.managerIds(managerIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public EddystonesRequestExecutor eTag(final String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (eddystoneUIDs != null) {
      params.put(CloudConstants.Devices.EDDYSTONE_UID_PARAMETER, StringUtils.join(eddystoneUIDs, ","));
    }
    if (eddystoneEIDs != null) {
      params.put(CloudConstants.Devices.EDDYSTONE_EID_PARAMETER, StringUtils.join(eddystoneEIDs, ","));
    }
    return params;
  }
}
