package com.kontakt.sdk.android.cloud.api.executor.commands;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.CommandsApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.CommandsService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.paginated.SecureCommands;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.model.SecureCommandType;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import retrofit2.Call;

/**
 * Request executor provided by {@link CommandsApi}. Use this class if you want to obtain
 * secure commands through fluent API in chained fashion. Here is an example of how to use this executor:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   SecureCommands commands = kontaktCloud.commands().fetch()
 *      .forDevices("34Gh", "5Gk9", "d5Gv")
 *      .withType(SecureCommandType.BOOTLOADER)
 *      .startIndex(5)
 *      .maxResult(15)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that devices and command type must be specified so invocations of {@code forDevices}
 * and {@code withType} methods are mandatory. Otherwise an exception will be thrown.
 */
public class CommandsRequestExecutor extends PaginatedRequestExecutor<SecureCommands> {

  private final CommandsService commandsService;

  private String[] uniqueIds;
  private SecureCommandType type;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param commandsService the commands API service.
   */
  public CommandsRequestExecutor(final CommandsService commandsService) {
    this.commandsService = commandsService;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CommandsRequestExecutor forDevices(final String... uniqueIds) {
    this.uniqueIds = SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    return this;
  }

  /**
   * Specifies devices. The method invocation is obligatory while using this request executor.
   *
   * @param uniqueIds the device unique identifiers.
   * @return this request executor.
   */
  public CommandsRequestExecutor forDevices(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies the command's type. The method invocation is obligatory while using this request executor.
   *
   * @param type the secure command type.
   * @return this request executor.
   */
  public CommandsRequestExecutor withType(final SecureCommandType type) {
    this.type = SDKPreconditions.checkNotNull(type, "type cannot be null");
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandsRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandsRequestExecutor maxResult(int maxResult) {
    super.startIndex(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandsRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandsRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandsRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SecureCommands execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(CloudCallback<SecureCommands> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<SecureCommands> prepareCall() {
    return commandsService.getSecureCommands(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(uniqueIds != null, "Cannot get commands - specify devices");
    SDKPreconditions.checkState(type != null, "Cannot get commands - specify command type");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    params.put(CloudConstants.Commands.COMMAND_PARAMETER, type.name());
    return params;
  }
}
