package com.kontakt.sdk.android.cloud.api.executor.activities;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActivitiesApi;
import com.kontakt.sdk.android.cloud.api.executor.PaginatedRequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActivitiesService;
import com.kontakt.sdk.android.cloud.response.paginated.Activities;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.OrderBy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Request executor provided by {@link ActivitiesApi}. Use this class if you want to fetch
 * activities through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Activities activities = kontaktCloud.activities().fetch()
 *      .startIndex(5)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class ActivitiesRequestExecutor extends PaginatedRequestExecutor<Activities> {

  private final ActivitiesService activitiesService;
  private final List<UUID> activitiesIds = new ArrayList<>();

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param service the activities API service.
   */
  public ActivitiesRequestExecutor(final ActivitiesService service) {
    this.activitiesService = service;
  }

  /**
   * Use that method if you want to fetch particular activities by ID.
   *
   * @param ids activities unique identifiers.
   * @return this request executor.
   */
  public ActivitiesRequestExecutor withIds(final UUID... ids) {
    return withIds(Arrays.asList(ids));
  }

  /**
   * Use that method if you want to fetch particular activities by ID.
   *
   * @param ids activities unique identifiers.
   * @return this request executor.
   */
  public ActivitiesRequestExecutor withIds(final List<UUID> ids) {
    checkNotNull(ids, "Ids are null");
    this.activitiesIds.addAll(ids);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActivitiesRequestExecutor startIndex(int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActivitiesRequestExecutor maxResult(int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActivitiesRequestExecutor filter(String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActivitiesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ActivitiesRequestExecutor eTag(String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Activities> prepareCall() {
    final Call<Activities> call;
    if (eTag != null) {
      call = activitiesService.getActivities(params(), eTag);
    } else {
      call = activitiesService.getActivities(params());
    }
    return call;
  }

  @Override
  protected Map<String, String> params() {
    if (activitiesIds.isEmpty()) {
      return super.params();
    }
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Activities.ACTIVITY_ID_PARAMETER, StringUtils.join(activitiesIds, ","));
    params.putAll(super.params());
    return params;
  }
}
