package com.kontakt.sdk.android.ble.security;

import java.nio.ByteBuffer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Random;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import static com.kontakt.sdk.android.ble.util.EncryptUtils.md5;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNullOrEmpty;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkState;

public class PayloadEncrypter {

  private static final String UTF_CHARSET = "UTF-8";
  private static final String ENCRYPTION_ALGORITHM = "AES/OFB/NoPadding";
  private static final String KEY_ALGORITHM = "AES";
  private static final int IV_LENGTH = 12; //Packet#getData puts first 4 bytes of IV (token part)
  private static final Random RANDOM = new Random();

  private final String password;
  private final IvParameterSpec ivParameterSpec;

  public PayloadEncrypter(String password, byte[] iv) {
    this.password = password;
    ivParameterSpec = new IvParameterSpec(iv);
  }

  public PayloadEncrypter(String password, int token) {
    this.password = password;
    try {
      byte[] iv = new byte[IV_LENGTH];
      SecureRandom.getInstance("SHA1PRNG").nextBytes(iv);
      ByteBuffer ivBuffer = ByteBuffer.allocate(16);
      ivBuffer.putInt(token);
      ivBuffer.put(iv);
      ivParameterSpec = new IvParameterSpec(ivBuffer.array());
    } catch (NoSuchAlgorithmException e) {
      throw new RuntimeException(e);
    }
  }

  public byte[] getRandomPartOfIV() {
    byte[] iv = ivParameterSpec.getIV();
    return Arrays.copyOfRange(iv, 4, 16);
  }

  public byte[] encrypt(byte[] data) {
    byte[] key = md5(password, UTF_CHARSET);
    SecretKeySpec keySpec = new SecretKeySpec(key, KEY_ALGORITHM);
    try {
      Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORITHM);
      cipher.init(Cipher.ENCRYPT_MODE, keySpec, ivParameterSpec);
      return cipher.doFinal(data);
    } catch (NoSuchAlgorithmException | NoSuchPaddingException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException | InvalidAlgorithmParameterException e) {
      throw new RuntimeException(e);
    }
  }

  public byte[] decrypt(byte[] data) {
    byte[] key = md5(password, UTF_CHARSET);
    SecretKeySpec keySpec = new SecretKeySpec(key, KEY_ALGORITHM);
    try {
      Cipher cipher = Cipher.getInstance(ENCRYPTION_ALGORITHM);
      cipher.init(Cipher.DECRYPT_MODE, keySpec, ivParameterSpec);
      return cipher.doFinal(data);
    } catch (IllegalBlockSizeException | InvalidKeyException | BadPaddingException | NoSuchPaddingException | NoSuchAlgorithmException | InvalidAlgorithmParameterException e) {
      throw new RuntimeException(e);
    }
  }

  public static byte[] encryptPayload(byte[] alignedData, String password, int token) {
    checkState(alignedData.length % 16 == 0, "Data not aligned. Cannot encrypt.");
    checkNotNullOrEmpty(password, "Password cannot be null.");
    byte[] crc = CRCModbus.calculateToBytes(alignedData);
    PayloadEncrypter encrypter = new PayloadEncrypter(password, token);
    int length = IV_LENGTH + alignedData.length + crc.length;
    ByteBuffer buffer = ByteBuffer.allocate(length);
    buffer.put(encrypter.getRandomPartOfIV());
    buffer.put(encrypter.encrypt(alignedData));
    buffer.put(crc);
    return buffer.array();
  }

  public static byte[] align(byte[] data, int mod, Byte paddingByte) {
    if (data.length % mod != 0) {
      int alignmentBytes = mod - (data.length % mod);
      ByteBuffer buf = ByteBuffer.allocate(data.length + alignmentBytes);
      buf.put(data);
      while (buf.hasRemaining()) {
        if (paddingByte == null) {
          buf.put((byte) (RANDOM.nextInt() & 0xFF));
        } else {
          buf.put(paddingByte);
        }
      }
      data = buf.array();
    }
    return data;
  }
}
