package com.kontakt.sdk.android.ble.discovery.eddystone;

import android.bluetooth.BluetoothDevice;
import android.util.SparseIntArray;
import com.kontakt.sdk.android.ble.configuration.ScanContext;
import com.kontakt.sdk.android.ble.device.EddystoneDevice;
import com.kontakt.sdk.android.ble.discovery.DiscoveryUtils;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.ble.discovery.ScanResponse;
import com.kontakt.sdk.android.ble.spec.EddystoneFrameType;
import com.kontakt.sdk.android.ble.spec.Telemetry;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.IEddystoneDevice;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.LimitedLinkedHashMap;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import static com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver.DEFAULT_INSTANCE_ID_START_INDEX;
import static com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver.DEFAULT_NAMESPACE_ID_START_INDEX;

@SuppressWarnings("MissingPermission")
public final class EddystoneParser extends Parser<EddystoneDevice> {

  private static final int HEADER_LENGTH = 12;

  private static final int HEADER_BLE_FLAGS_LENGTH_OF_FLAGS_BLOCK = 0x02;
  private static final int HEADER_BLE_FLAGS_PDU_DATA_TYPE = 0x01;
  private static final int HEADER_BLE_FLAGS_DATA = 0x06;
  private static final int HEADER_SERVICE_UUID_LENGTH_OF_SERVICE_UUID_BLOCK = 0x03;
  private static final int HEADER_SERVICE_UUID_PDU_DATA_TYPE = 0x03;

  private static final byte HEADER_SERVICE_UUID_HIGH_ORDER_VALUE = (byte) 0xAA;
  private static final byte HEADER_SERVICE_UUID_LOW_ORDER_VALUE = (byte) 0xFE;
  private static final int EDDYSTONE_PACKET_TX_POWER_INDEX = 3;

  private static final int EDDYSTONE_ENCRYPTED_TLM_VERSION = 0x01;

  private static final NamespaceIdResolver NAMESPACE_ID_RESOLVER = new NamespaceIdResolver(DEFAULT_NAMESPACE_ID_START_INDEX);
  private static final InstanceIdResolver INSTANCE_ID_RESOLVER = new InstanceIdResolver(DEFAULT_INSTANCE_ID_START_INDEX);
  private static final URLResolver URL_RESOLVER = new URLResolver();
  private static final TLMResolver TLM_RESOLVER = new TLMResolver();
  private static final ETLMResolver ETLM_RESOLVER = new ETLMResolver();
  private static final EIDResolver EID_RESOLVER = new EIDResolver();

  private static byte[] EDDYSTONE_SPECIFIC_HEADER = new byte[] {
      HEADER_BLE_FLAGS_LENGTH_OF_FLAGS_BLOCK, HEADER_BLE_FLAGS_PDU_DATA_TYPE, HEADER_BLE_FLAGS_DATA, HEADER_SERVICE_UUID_LENGTH_OF_SERVICE_UUID_BLOCK,
      HEADER_SERVICE_UUID_PDU_DATA_TYPE, HEADER_SERVICE_UUID_HIGH_ORDER_VALUE, HEADER_SERVICE_UUID_LOW_ORDER_VALUE
  };

  private final Map<String, ScanResponse> scanResponseCache = new LimitedLinkedHashMap<>(CACHE_SIZE);
  private final SparseIntArray txPowerCache = new SparseIntArray();
  private final Set<IEddystoneNamespace> namespaces;
  private final Collection<EddystoneFrameType> triggerFrameTypes;

  public EddystoneParser(final ScanContext scanContext) {
    super(scanContext);
    this.namespaces = scanContext.getEddystoneNamespaces();
    this.triggerFrameTypes = scanContext.getEddystoneFrameTypes();
  }

  boolean isValidEddystoneFrame(final byte[] scanRecord) {
    return !(scanRecord == null || scanRecord.length < HEADER_LENGTH) &&
        ConversionUtils.doesArrayBeginWith(scanRecord, EDDYSTONE_SPECIFIC_HEADER) &&
        scanRecord[9] == HEADER_SERVICE_UUID_HIGH_ORDER_VALUE &&
        scanRecord[10] == HEADER_SERVICE_UUID_LOW_ORDER_VALUE;
  }

  public boolean parseFrame(EddystoneFrameType frameType, String deviceAddress, byte[] scanRecord) {
    frameData.clear();
    extractFrameData(scanRecord, frameData);
    byte[] packetFrame = frameData.get(FrameDataType.EDDYSTONE_PACKET_SERVICE_DATA);
    if (packetFrame == null) {
      return false;
    }
    saveScanResponse(deviceAddress);
    switch (frameType) {
      case UID:
        return saveUID(deviceAddress, packetFrame);
      case URL:
        return saveURL(deviceAddress, packetFrame);
      case TLM:
        return isTlmPacketEncrypted(packetFrame) ? saveETLM(deviceAddress, packetFrame) : saveTLM(deviceAddress, packetFrame);
      case EID:
        return saveEID(deviceAddress, packetFrame);
      default:
        Logger.e("Unknown Eddystone packetFrame type parsed for device with address: " + deviceAddress);
        return false;
    }
  }

  private void saveScanResponse(final String deviceAddress) {
    boolean isScanResponsePresent = isScanResponsePresent();
    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    if (!scanResponseCache.containsKey(deviceAddress)) {
      ScanResponse scanResponse = isScanResponsePresent ? ScanResponse.fromScanResponseBytes(serviceData) : ScanResponse.UNKNOWN;
      scanResponseCache.put(deviceAddress, scanResponse);
    } else {
      if (scanResponseCache.get(deviceAddress).isUnknown() && isScanResponsePresent) {
        scanResponseCache.put(deviceAddress, ScanResponse.fromScanResponseBytes(serviceData));
      }
    }
  }

  private boolean saveUID(String deviceAddress, byte[] frame) {
    String namespace = NAMESPACE_ID_RESOLVER.parse(frame);
    String instanceId = INSTANCE_ID_RESOLVER.parse(frame);

    if (namespace == null || instanceId == null) {
      return false;
    }

    updateTxPower(deviceAddress, frame[EDDYSTONE_PACKET_TX_POWER_INDEX]);
    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().namespace(namespace).instanceId(instanceId).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setNamespace(namespace);
      device.setInstanceId(instanceId);
    }
    return true;
  }

  private boolean saveURL(String deviceAddress, byte[] frame) {
    String url = URL_RESOLVER.parse(frame);

    if (url == null) {
      return false;
    }

    updateTxPower(deviceAddress, frame[EDDYSTONE_PACKET_TX_POWER_INDEX]);
    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().url(url).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setUrl(url);
    }
    return true;
  }

  private boolean saveTLM(String deviceAddress, byte[] frame) {
    Telemetry telemetry = TLM_RESOLVER.parse(frame);

    if (telemetry == null) {
      return false;
    }

    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().telemetry(telemetry).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setTelemetry(telemetry);
    }
    return true;
  }

  private boolean saveETLM(String deviceAddress, byte[] frame) {
    String etlm = ETLM_RESOLVER.parse(frame);

    if (etlm == null) {
      return false;
    }

    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().encryptedTelemetry(etlm).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setEncryptedTelemetry(etlm);
    }
    return true;
  }

  private boolean saveEID(String deviceAddress, byte[] frame) {
    String eid = EID_RESOLVER.parse(frame);

    if (eid == null) {
      return false;
    }

    updateTxPower(deviceAddress, frame[EDDYSTONE_PACKET_TX_POWER_INDEX]);
    EddystoneDevice device = getCachedEddystoneDevice(deviceAddress);
    if (device == null) {
      device = new EddystoneDevice.Builder().eid(eid).build();
      putEddystoneDeviceInCache(deviceAddress, device);
    } else {
      device.setEid(eid);
    }
    return true;
  }

  IEddystoneDevice getEddystoneDevice(BluetoothDevice bluetoothDevice, int rssi) {
    String deviceAddress = bluetoothDevice.getAddress();

    int txPower = getTxPower(deviceAddress.hashCode());
    int calculatedRssi = rssiCalculator.calculateRssi(deviceAddress.hashCode(), rssi);
    double distance = DiscoveryUtils.calculateDistance(txPower, calculatedRssi, DeviceProfile.EDDYSTONE);
    Proximity proximity = Proximity.fromDistance(distance);

    String name = bluetoothDevice.getName();
    ScanResponse scanResponse = getScanResponse(deviceAddress);

    IEddystoneDevice cachedEddystoneDevice = getCachedEddystoneDevice(deviceAddress);
    EddystoneDevice device = new EddystoneDevice.Builder(cachedEddystoneDevice).address(deviceAddress)
        .name(name)
        .uniqueId(scanResponse.getUniqueId())
        .firmwareRevision(scanResponse.getFirmwareVersion())
        .batteryPower(scanResponse.getBatteryPower())
        .shuffled(isShuffled(scanResponse, getNamespaceForDevice(deviceAddress)))
        .txPower(txPower)
        .distance(distance)
        .proximity(proximity)
        .rssi(calculatedRssi)
        .timestamp(System.currentTimeMillis())
        .build();

    putEddystoneDeviceInCache(deviceAddress, device);

    return device;
  }

  private boolean isShuffled(ScanResponse scanResponse, String namespace) {
    return scanResponse.isUnknown() ? isDefinedSecureNamespace(namespace) : scanResponse.isShuffled();
  }

  private boolean isDefinedSecureNamespace(String namespace) {
    if (namespace == null) {
      return false;
    }
    for (IEddystoneNamespace definedNamespace : namespaces) {
      if (definedNamespace.getSecureNamespace() != null && namespace.equalsIgnoreCase(definedNamespace.getSecureNamespace())) {
        return true;
      }
    }
    return false;
  }

  private String getNamespaceForDevice(String deviceAddress) {
    return getCachedEddystoneDevice(deviceAddress).getNamespace();
  }

  boolean areTriggerFramesParsed(String deviceAddress) {
    if (triggerFrameTypes.isEmpty()) {
      return true;
    }

    IEddystoneDevice eddystone = getCachedEddystoneDevice(deviceAddress);

    final boolean isUIDNotCached = (!hasFrame(eddystone, EddystoneFrameType.UID));
    if (triggerFrameTypes.contains(EddystoneFrameType.UID) && isUIDNotCached) {
      return false;
    }

    final boolean isURLNotCached = (!hasFrame(eddystone, EddystoneFrameType.URL));
    if (triggerFrameTypes.contains(EddystoneFrameType.URL) && isURLNotCached) {
      return false;
    }

    final boolean isTLMNotCached = (!hasFrame(eddystone, EddystoneFrameType.TLM));
    if (triggerFrameTypes.contains(EddystoneFrameType.TLM) && isTLMNotCached) {
      return false;
    }

    final boolean isEIDNotCached = (!hasFrame(eddystone, EddystoneFrameType.EID));
    if (triggerFrameTypes.contains(EddystoneFrameType.EID) && isEIDNotCached) {
      return false;
    }

    return true;
  }

  private ScanResponse getScanResponse(String deviceAddress) {
    return scanResponseCache.get(deviceAddress);
  }

  int getTxPowerCacheSize() {
    return txPowerCache.size();
  }

  private int getTxPower(int deviceAddressHashCode) {
    return txPowerCache.get(deviceAddressHashCode);
  }

  private void putEddystoneDeviceInCache(String deviceAddress, EddystoneDevice device) {
    int hashCode = hashCodeBuilder.append(deviceAddress).build();
    devicesCache.put(hashCode, device);
  }

  EddystoneDevice getCachedEddystoneDevice(String deviceAddress) {
    int hashCode = hashCodeBuilder.append(deviceAddress).build();
    return devicesCache.get(hashCode);
  }

  private void updateTxPower(final String deviceAddress, final int txPower) {
    txPowerCache.put(deviceAddress.hashCode(), txPower);
  }

  boolean hasFrame(IEddystoneDevice eddystone, EddystoneFrameType frameType) {
    if (eddystone == null || frameType == null) {
      return false;
    }
    switch (frameType) {
      case UID:
        return eddystone.getNamespace() != null && eddystone.getInstanceId() != null;
      case URL:
        return eddystone.getUrl() != null;
      case TLM:
        return eddystone.getTelemetry() != null || eddystone.getEncryptedTelemetry() != null;
      case EID:
        return eddystone.getEid() != null;
    }
    return false;
  }

  private boolean isScanResponsePresent() {
    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    return serviceData != null && ScanResponse.isValidKontaktScanResponse(serviceData);
  }

  private boolean isTlmPacketEncrypted(byte[] tlmPacketFrame) {
    return tlmPacketFrame[3] == EDDYSTONE_ENCRYPTED_TLM_VERSION;
  }

  @Override
  protected void disable() {
    if (isEnabled) {
      isEnabled = false;
      devicesCache.clear();
      rssiCalculator.clear();
      scanResponseCache.clear();
      txPowerCache.clear();
    }
  }
}
