package com.kontakt.sdk.android.ble.connection;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattService;
import android.os.Build;
import android.os.RemoteException;
import com.kontakt.sdk.android.ble.exception.CharacteristicAbsentException;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceService;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * Acts as a container of characteristics requested before authorization.
 */
public class KontaktDeviceServiceStore {

  private static final KontaktDeviceService REQUIRED_SERVICE_NORMAL_MODE = KontaktDeviceService.CONTROL;
  private static final KontaktDeviceService REQUIRED_SERVICE_SECURE_MODE = KontaktDeviceService.SECURE_CONFIGURATION_SERVICE;

  private final Map<KontaktDeviceService, Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic>> storeMap;
  private final StoreType storeType;

  /**
   * Type of store
   * <ul>
   * <li>{@link #NORMAL} if beacon firmware revision is lower than 4.0</li>
   * <li>{@link #SECURE} if beacon firmware revision is 4.0 and greater</li>
   * </ul>
   */
  public enum StoreType {
    NORMAL,
    SECURE
  }

  /**
   * Instantiates a new Service store. If services sent are not compatible with
   * the ones propagated by kontakt.io beacons(services list does not contain
   * all required services: Service 1, Service 2, Service 3, Service 4,
   * Service 5, Service 6, Service 7) then the RemoteException (informing that
   * the beacon is not a property of kontakt.io) is thrown.
   *
   * @param gattServices beacon services list
   * @param storeType store type
   * @throws RemoteException the remote exception thrown if services list does
   * not match the specification of kontakt.io beacon
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public KontaktDeviceServiceStore(final List<BluetoothGattService> gattServices, StoreType storeType) throws RemoteException {
    SDKPreconditions.checkNotNullOrEmpty(gattServices, "Gatt Service list is null.");
    this.storeMap = new HashMap<>(gattServices.size());
    this.storeType = storeType;
    createStore(gattServices);
    try {
      assertRequiredServiceFound(storeType);
      assertRequiredServicesFoundIfDfuModeEnabled();
    } catch (ServiceAbsentException e) {
      throw new RemoteException("Beacon is not recognized as a Kontakt.io device");
    }
  }

  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  private void createStore(List<BluetoothGattService> gattServices) {
    for (BluetoothGattService gattService : gattServices) {
      KontaktDeviceService service = KontaktDeviceService.valueOf(gattService.getUuid());
      if (service == null) {
        continue;
      }

      Collection<BluetoothGattCharacteristic> gattCharacteristics = gattService.getCharacteristics();
      Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> characteristics = storeMap.get(service);
      if (characteristics == null) {
        characteristics = new HashMap<>(gattCharacteristics.size());
        storeMap.put(service, characteristics);
      }

      for (BluetoothGattCharacteristic gattCharacteristic : gattCharacteristics) {
        KontaktDeviceCharacteristic characteristic = KontaktDeviceCharacteristic.fromUuid(gattCharacteristic.getUuid());
        if (characteristic == null) {
          continue;
        }
        characteristics.put(characteristic, new BluetoothDeviceCharacteristic(gattCharacteristic));
      }
    }
  }

  /**
   * Gets firmware revision wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getFirmwareRevisionCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.DEVICE_INFORMATION), KontaktDeviceCharacteristic.FIRMWARE_REVISION_STRING);
  }

  /**
   * Gets device time wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getDeviceTimeCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CURRENT_TIME), KontaktDeviceCharacteristic.CURRENT_TIME);
  }

  /**
   * Gets hardware revision wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getHardwareRevisionCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.DEVICE_INFORMATION), KontaktDeviceCharacteristic.HARDWARE_REVISION);
  }

  /**
   * Gets manufacturer name wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getManufacturerNameCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.DEVICE_INFORMATION), KontaktDeviceCharacteristic.MANUFACTURER_NAME);
  }

  /**
   * Gets device name wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getDeviceNameCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.GENERIC_ACCESS), KontaktDeviceCharacteristic.DEVICE_NAME);
  }

  /**
   * Gets propagated device name wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getPropagatedDeviceNameCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.PROPAGATED_DEVICE_NAME);
  }

  /**
   * Gets minor wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getMinorCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.MINOR);
  }

  /**
   * Gets non connectable wrapper.
   * The Non-connectable characteristic was added with "2.6" firmware. If beacon
   * has lower firmware, then it returns null.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getNonConnectableCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.NON_CONNECTABLE_MODE);
  }

  /**
   * Provides shuffle interval wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getShuffleIntervalCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.SHUFFLE_INTERVAL);
  }

  /**
   * Provides shuffle key wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getShuffleKeyCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.SHUFFLE_KEY);
  }

  /**
   * Provides sensor on wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getSensorsCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.SENSORS_SERVICE), KontaktDeviceCharacteristic.SENSOR_ON);
  }

  /**
   * Provides light sensor percentage wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getLightSensorCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.KONTAKT_SENSORS_SERVICE), KontaktDeviceCharacteristic.LIGHT_SENSOR_PERCENTAGE);
  }

  /**
   * Provides accelerometer wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getAccelerometerCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.SENSORS_SERVICE), KontaktDeviceCharacteristic.ACCELEROMETER);
  }

  /**
   * Gets active profile wrapper.
   * The active profile characteristic wrapper. The wrapper was added in "3.0" firmware version.
   * If beacon has lower firmware, then null is returned.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getActiveProfileCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.ACTIVE_PROFILE);
  }

  /**
   * Gets major wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getMajorCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.MAJOR);
  }

  /**
   * Gets proximity wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getProximityCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.PROXIMITY_UUID);
  }

  /**
   * Gets power level wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getPowerLevelCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.TX_POWER), KontaktDeviceCharacteristic.TX_POWER_LEVEL);
  }

  /**
   * Gets battery level wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getBatteryLevelCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.BATTERY_LEVEL), KontaktDeviceCharacteristic.BATTERY_LEVEL);
  }

  /**
   * Gets password wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getPasswordCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.PASSWORD);
  }

  /**
   * Provides master password wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getMasterPasswordCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.MASTER_PASSWORD);
  }

  /**
   * Gets set new password wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getSetNewPasswordCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.SET_PASSWORD);
  }

  /**
   * Gets advertising interval wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getAdvertisingIntervalCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.TIMERS), KontaktDeviceCharacteristic.ADVERTISING_INTERVAL);
  }

  /**
   * Gets reset wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getResetCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.RESET);
  }

  /**
   * Gets bootloader wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getBootloaderCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.BOOTLOADER);
  }

  /**
   * Provides unique id wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getUniqueIdCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.BEACON_ID);
  }

  /**
   * Provides namespace id wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getNamespaceIdCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.NAMESPACE_ID);
  }

  /**
   * Provides instance id wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getInstanceIdCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.INSTANCE_ID);
  }

  /**
   * Provides url wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getUrlCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.PROXIMITY), KontaktDeviceCharacteristic.URL_ID);
  }

  /**
   * Gets default settings wrapper.
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getDefaultSettingsCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.CONTROL), KontaktDeviceCharacteristic.DEFAULT_SETTINGS);
  }

  /**
   * Gets last processed requet token wrapper
   * <p>
   * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getSecureLastProcessedRequestTokenCharacteristic()
      throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.SECURE_CONFIGURATION_SERVICE),
        KontaktDeviceCharacteristic.SECURE_LAST_PROCESSED_REQUEST_TOKEN);
  }

  /**
   * Gets secure write wrapper
   * <p>
   * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public BluetoothDeviceCharacteristic getSecureWriteCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    BluetoothDeviceCharacteristic characteristic =
        getCharacteristic(getService(KontaktDeviceService.SECURE_CONFIGURATION_SERVICE), KontaktDeviceCharacteristic.SECURE_WRITE);
    characteristic.setWriteType(BluetoothGattCharacteristic.WRITE_TYPE_DEFAULT);//Required after Beacon Pro 1.8 firmware was introduced
    return characteristic;
  }

  /**
   * Gets secure write response wrapper
   * <p>
   * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getSecureResponseCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.SECURE_CONFIGURATION_SERVICE), KontaktDeviceCharacteristic.SECURE_RESPONSE);
  }

  /**
   * Gets secure control point wrapper
   * <p>
   * Characteristic added in firmware revision 4.0, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getSecureControlPointCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.SECURE_CONFIGURATION_SERVICE), KontaktDeviceCharacteristic.SECURE_CONTROL_POINT);
  }

  /**
   * Gets Kontakt DFU command wrapper
   * <p>
   * Characteristic added in firmware revision 1.0 of Beacon PRO devices, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getDfuCommandCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.KONTAKT_DFU_SERVICE), KontaktDeviceCharacteristic.KDFU_COMMAND);
  }

  /**
   * Gets Kontakt DFU response wrapper
   * <p>
   * Characteristic added in firmware revision 1.0 of Beacon PRO devices, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getDfuResponseCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.KONTAKT_DFU_SERVICE), KontaktDeviceCharacteristic.KDFU_RESPONSE);
  }

  /**
   * Gets Kontakt DFU data wrapper
   * <p>
   * Characteristic added in firmware revision 1.0 of Beacon PRO devices, on lower firmware revision throws exception
   *
   * @return {@link BluetoothDeviceCharacteristic}  characteristic wrapper
   * @throws ServiceAbsentException if service not found
   * @throws CharacteristicAbsentException if characteristic not found
   */
  public BluetoothDeviceCharacteristic getDfuDataCharacteristic() throws ServiceAbsentException, CharacteristicAbsentException {
    return getCharacteristic(getService(KontaktDeviceService.KONTAKT_DFU_SERVICE), KontaktDeviceCharacteristic.KDFU_DATA);
  }

  /**
   * Gets type of store
   *
   * @return {@link com.kontakt.sdk.android.ble.connection.KontaktDeviceServiceStore.StoreType}
   */
  public StoreType getStoreType() {
    return storeType;
  }

  /**
   * Checks if {@link KontaktDeviceService} was found during service discovery
   *
   * @param service model to find
   * @return true or false
   */
  public boolean contains(KontaktDeviceService service) {
    return storeMap.containsKey(service);
  }

  /**
   * Replaces provided characteristic in cache
   *
   * @param characteristicWrapper new characteristic
   * @throws ServiceAbsentException if service not found
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)

  public void replace(BluetoothDeviceCharacteristic characteristicWrapper) throws ServiceAbsentException {
    UUID serviceUuid = characteristicWrapper.getService().getUuid();
    UUID characteristicUuid = characteristicWrapper.getId();

    KontaktDeviceService serviceModel = KontaktDeviceService.valueOf(serviceUuid);
    KontaktDeviceCharacteristic characteristicModel = KontaktDeviceCharacteristic.fromUuid(characteristicUuid);

    storeMap.get(serviceModel).put(characteristicModel, characteristicWrapper);
  }

  /**
   * Clears cache
   */
  public void clear() {
    storeMap.clear();
  }

  /**
   * Gets all readable characteristic found during service discovery
   *
   * @return Collection of readable characteristic
   */
  public final Collection<BluetoothDeviceCharacteristic> getReadableCharacteristics() {

    Collection<BluetoothDeviceCharacteristic> readableCharacteristics = new ArrayList<BluetoothDeviceCharacteristic>();

    for (Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> characteristicMap : storeMap.values()) {
      for (BluetoothDeviceCharacteristic characteristic : characteristicMap.values()) {
        if (characteristic.isReadable()) {
          readableCharacteristics.add(characteristic);
        }
      }
    }

    return Collections.unmodifiableCollection(readableCharacteristics);
  }

  Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> getService(KontaktDeviceService service) throws ServiceAbsentException {
    SDKPreconditions.checkNotNull(service, "Service is null.");
    SDKPreconditions.checkArgument(contains(service),
        new ServiceAbsentException(String.format("The Service %s was not found during services discovery", service)));

    return Collections.unmodifiableMap(storeMap.get(service));
  }

  BluetoothDeviceCharacteristic getCharacteristic(Map<KontaktDeviceCharacteristic, BluetoothDeviceCharacteristic> map,
      KontaktDeviceCharacteristic characteristic) throws CharacteristicAbsentException {
    SDKPreconditions.checkState(map.containsKey(characteristic),
        new CharacteristicAbsentException(String.format("The characteristic %s is absent", characteristic == null ? "" : characteristic)));
    return map.get(characteristic);
  }

  private void assertRequiredServiceFound(StoreType storeType) throws ServiceAbsentException {
    switch (storeType) {
      case NORMAL:
        getService(REQUIRED_SERVICE_NORMAL_MODE);
        break;
      case SECURE:
        getService(REQUIRED_SERVICE_SECURE_MODE);
        break;
    }
  }

  private void assertRequiredServicesFoundIfDfuModeEnabled() throws ServiceAbsentException {
    if (contains(KontaktDeviceService.DFU_SERVICE)) {
      final Set<KontaktDeviceService> requiredServices =
          EnumSet.of(KontaktDeviceService.GENERIC_ACCESS, KontaktDeviceService.GENERIC_ATTRIBUTE, KontaktDeviceService.DFU_SERVICE);

      for (KontaktDeviceService kontaktDeviceService : requiredServices) {
        getService(kontaktDeviceService);
      }
    }
  }
}
