package com.kontakt.sdk.android.ble.connection;

import android.text.TextUtils;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Config;
import com.kontakt.sdk.android.common.model.Preset;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;
import com.kontakt.sdk.android.common.util.Constants;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * Write batch processor writes prepared sets of arguments to device.
 * The batch is represented by Configs ({@link Config})
 * and Profiles ({@link Preset}).
 */
class KontaktDeviceBatchProcessor {

  private static final String TAG = KontaktDeviceBatchProcessor.class.getSimpleName();

  private final Batch batch;
  final KontaktDeviceConnectionImpl kontaktDeviceConnection;

  Map.Entry<String, Object> currentEntry = null;

  /**
   * Creates Write Batch processor instance.
   *
   * @param batch selected batch instance
   * @param connection the BeaconConnection
   * @return the WriteBatchProcessor instance
   */
  static KontaktDeviceBatchProcessor from(final Batch batch, final KontaktDeviceConnectionImpl connection) {
    return new KontaktDeviceBatchProcessor(batch, connection);
  }

  private KontaktDeviceBatchProcessor(final Batch batch, final KontaktDeviceConnectionImpl connection) {
    this.batch = batch;
    this.kontaktDeviceConnection = connection;
  }

  /**
   * Performs writing parameters batch to Beacon device.
   *
   * @param processingListener the processing listener
   */
  void process(final ProcessingListener processingListener) {
    processingListener.onStart();
    final Iterator<Map.Entry<String, Object>> argumentIterator = batch.processIterator();

    if (batch.isProcessEmpty()) {
      kontaktDeviceConnection.finishWriteBatch();
      processingListener.onFinish();
      return;
    }

    if (argumentIterator.hasNext()) {
      process(argumentIterator, processingListener);
    }
  }

  void writeCheckNextArgument(Iterator<Map.Entry<String, Object>> argumentIterator, final ProcessingListener processingListener) {
    if (argumentIterator.hasNext()) {
      argumentIterator.remove();
      process(argumentIterator, processingListener);
    } else {
      kontaktDeviceConnection.finishWriteBatch();
      processingListener.onFinish();
    }
  }

  void writeFailure(final ProcessingListener processingListener) {
    final int errorCode = getErrorCode(currentEntry.getKey());
    Logger.e(TAG + ": write operation finished with failure - error code = " + errorCode);
    processingListener.onError(getErrorCode(currentEntry.getKey()));
    performRollback(processingListener);
  }

  private void process(final Iterator<Map.Entry<String, Object>> argumentIterator, final ProcessingListener processingListener) {

    final WriteListener nextWriteListener = new WriteListener() {
      @Override
      public void onWriteSuccess(WriteResponse response) {
        writeCheckNextArgument(argumentIterator, processingListener);
      }

      @Override
      public void onWriteFailure(ErrorCause cause) {
        if (ErrorCause.FEATURE_NOT_SUPPORTED == cause) {
          writeCheckNextArgument(argumentIterator, processingListener);
        } else {
          writeFailure(processingListener);
        }
      }
    };

    currentEntry = argumentIterator.next();
    final String key = currentEntry.getKey();

    switch (key) {
      case Constants.Eddystone.URL:
        kontaktDeviceConnection.overwriteUrl((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Eddystone.INSTANCE_ID:
        kontaktDeviceConnection.overwriteInstanceId((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Eddystone.NAMESPACE:
        kontaktDeviceConnection.overwriteNamespaceId((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.PROXIMITY:
        kontaktDeviceConnection.overwriteProximityUUID((UUID) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.INTERVAL:
        kontaktDeviceConnection.overwriteAdvertisingInterval((Long) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.TX_POWER:
        kontaktDeviceConnection.overwritePowerLevel((Integer) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.MAJOR:
        kontaktDeviceConnection.overwriteMajor((Integer) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.MINOR:
        kontaktDeviceConnection.overwriteMinor((Integer) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.PROFILES:
        kontaktDeviceConnection.switchToDeviceProfile((DeviceProfile) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.NAME:
        kontaktDeviceConnection.overwriteModelName((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.PASSWORD:
        kontaktDeviceConnection.overwritePassword((String) currentEntry.getValue(), nextWriteListener);
        break;
      default:
        throw new IllegalStateException("Unrecognised tag: " + key);
    }
  }

  private void performRollback(final ProcessingListener processingListener) {
    batch.notifyProcessChanged();
    final Iterator<Map.Entry<String, Object>> rollbackIterator = batch.rollbackIterator();
    if (rollbackIterator.hasNext()) {
      processingListener.onRollbackStart();
      processRollback(rollbackIterator, processingListener);
    }
  }

  void processRollback(final Iterator<Map.Entry<String, Object>> argumentIterator, final ProcessingListener processingListener) {

    final WriteListener nextWriteListener = new WriteListener() {
      @Override
      public void onWriteSuccess(WriteResponse response) {
        if (argumentIterator.hasNext()) {
          argumentIterator.remove();
          processRollback(argumentIterator, processingListener);
        } else {
          kontaktDeviceConnection.finishWriteBatch();
          processingListener.onRollbackFinish();
        }
      }

      @Override
      public void onWriteFailure(ErrorCause cause) {
        Logger.d("Batch not fully applied. Next write operation is required.");
        kontaktDeviceConnection.finishWriteBatch();
        processingListener.onRollbackError(getErrorCode(currentEntry.getKey()));
      }
    };

    currentEntry = argumentIterator.next();
    final String key = currentEntry.getKey();
    switch (currentEntry.getKey()) {
      case Constants.Devices.PROXIMITY:
        kontaktDeviceConnection.overwriteProximityUUID((UUID) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.INTERVAL:
        kontaktDeviceConnection.overwriteAdvertisingInterval((Long) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.TX_POWER:
        kontaktDeviceConnection.overwritePowerLevel((Integer) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.MAJOR:
        kontaktDeviceConnection.overwriteMajor((Integer) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.MINOR:
        kontaktDeviceConnection.overwriteMinor((Integer) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.PROFILES:
        kontaktDeviceConnection.switchToDeviceProfile((DeviceProfile) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Devices.NAME:
        kontaktDeviceConnection.overwriteModelName((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Eddystone.NAMESPACE:
        kontaktDeviceConnection.overwriteNamespaceId((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Eddystone.INSTANCE_ID:
        kontaktDeviceConnection.overwriteInstanceId((String) currentEntry.getValue(), nextWriteListener);
        break;
      case Constants.Eddystone.URL:
        kontaktDeviceConnection.overwriteUrl((String) currentEntry.getValue(), nextWriteListener);
        break;
      default:
        throw new IllegalStateException("Unrecognised tag: " + key);
    }
  }

  static int getErrorCode(final String key) {
    if (key.equals(Constants.Devices.INTERVAL)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_INTERVAL;
    } else if (key.equals(Constants.Devices.PROXIMITY)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_PROXIMITY_UUID;
    } else if (key.equals(Constants.Devices.TX_POWER)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_TX_POWER;
    } else if (key.equals(Constants.Devices.MAJOR)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_MAJOR;
    } else if (key.equals(Constants.Devices.MINOR)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_MINOR;
    } else if (key.equals(Constants.Devices.PROFILES)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_PROFILE;
    } else if (key.equals(Constants.Eddystone.NAMESPACE)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_NAMESPACE;
    } else if (key.equals(Constants.Eddystone.INSTANCE_ID)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_INSTANCE_ID;
    } else if (key.equals(Constants.Eddystone.URL)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_URL;
    } else if (key.equals(Constants.Devices.NAME)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_NAME;
    } else if (key.equals(Constants.Devices.PASSWORD)) {
      return DeviceConnectionError.ERROR_BATCH_WRITE_PASSWORD;
    } else {
      throw new IllegalStateException("Unrecognised tag: " + key);
    }
  }

  /**
   * Processing listener provides callback methods for Batch write process.
   */
  interface ProcessingListener {
    /**
     * Informs about Batch write process start.
     */
    void onStart();

    /**
     * Informs about error occured during Batch write.
     *
     * @param errorCode the error code
     */
    void onError(final int errorCode);

    /**
     * Informs about Batch write process finish.
     */
    void onFinish();

    /**
     * Informs about Rollback process start.
     */
    void onRollbackStart();

    /**
     * Informs about Rollback process finish.
     */
    void onRollbackFinish();

    /**
     * Informs about error occured during perforiming Rollback operation.
     *
     * @param errorCode the error code
     */
    void onRollbackError(final int errorCode);
  }

  /**
   * Batch represents selected parameters from Batch holder.
   * Only parameters different from
   */
  static class Batch {
    private final Map<String, Object> PROCESS_ARGUMENTS = new HashMap<>();
    private final Map<String, Object> ROLLBACK_ARGUMENTS = new HashMap<>();

    void notifyProcessChanged() {
      ROLLBACK_ARGUMENTS.entrySet().removeAll(PROCESS_ARGUMENTS.entrySet());
    }

    Iterator<Map.Entry<String, Object>> processIterator() {
      return PROCESS_ARGUMENTS.entrySet().iterator();
    }

    Iterator<Map.Entry<String, Object>> rollbackIterator() {
      return ROLLBACK_ARGUMENTS.entrySet().iterator();
    }

    boolean isProcessEmpty() {
      return PROCESS_ARGUMENTS.isEmpty();
    }

    /**
     * Select parameters batch from Config. Created batch consists of
     * parameters present in Config ({@link Config})
     * and different from current Beacon configuration.
     *
     * @param config the Config instance
     * @param characteristics the characteristics storing current Beacon configuration
     * @return new parameters Batch instance
     */
    static Batch select(final Config config, final RemoteBluetoothDevice.Characteristics characteristics) {
      SDKPreconditions.checkNotNull(config, "Config is null");

      final Batch batch = new Batch();

      String currentName = characteristics.getModelName();
      String desiredName = config.getName();
      if (!TextUtils.isEmpty(desiredName) && !currentName.equals(desiredName)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.NAME, desiredName);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.NAME, currentName);
      }

      final List<DeviceProfile> deviceProfile = config.getProfiles();
      if (deviceProfile != null && !deviceProfile.isEmpty()) {
        DeviceProfile desiredProfile = deviceProfile.get(0);
        DeviceProfile activeProfile = characteristics.getActiveProfile();
        if (activeProfile != null && desiredProfile != activeProfile) {
          batch.PROCESS_ARGUMENTS.put(Constants.Devices.PROFILES, desiredProfile);
          batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.PROFILES, activeProfile);
        }
      }

      final UUID proximityUUID = config.getProximity();
      final UUID characteristicProximityUUID = characteristics.getProximityUUID();
      if (proximityUUID != null && characteristicProximityUUID != null && !characteristics.getProximityUUID().equals(proximityUUID)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.PROXIMITY, proximityUUID);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.PROXIMITY, characteristicProximityUUID);
      }

      final int major = config.getMajor();
      if (major >= 0 && major != characteristics.getMajor() && characteristics.getMajor() > -1) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.MAJOR, major);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.MAJOR, characteristics.getMajor());
      }

      final int minor = config.getMinor();
      if (minor >= 0 && minor != characteristics.getMinor() && characteristics.getMinor() > -1) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.MINOR, minor);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.MINOR, characteristics.getMinor());
      }

      final long interval = config.getInterval();
      if (interval != characteristics.getAdvertisingInterval() && interval > 0) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.INTERVAL, interval);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.INTERVAL, characteristics.getAdvertisingInterval());
      }

      final int txPower = config.getTxPower();
      if (txPower != characteristics.getPowerLevel() && txPower > -1) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.TX_POWER, txPower);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.TX_POWER, characteristics.getPowerLevel());
      }

      final String namespace = config.getNamespace();
      final String characteristicNamespace = characteristics.getNamespaceId();
      if (!TextUtils.isEmpty(namespace) && !TextUtils.equals(namespace, characteristicNamespace)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Eddystone.NAMESPACE, namespace);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Eddystone.NAMESPACE, characteristicNamespace);
      }

      final String instanceId = config.getInstanceId();
      final String characteristicInstanceId = characteristics.getInstanceId();
      if (!TextUtils.isEmpty(instanceId) && !TextUtils.equals(instanceId, characteristicInstanceId)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Eddystone.INSTANCE_ID, instanceId);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Eddystone.INSTANCE_ID, characteristicInstanceId);
      }

      final String url = config.getUrl();
      final String characteristicUrl = characteristics.getUrl();
      if (!TextUtils.isEmpty(url) && !TextUtils.equals(url, characteristicUrl)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Eddystone.URL, url);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Eddystone.URL, characteristicUrl);
      }

      String devicePassword = config.getPassword();
      if (!TextUtils.isEmpty(devicePassword)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.PASSWORD, devicePassword);
      }

      return batch;
    }

    /**
     * Select parameters batch from Preset. Created batch
     * consists of parameters present in Preset
     * ({@link Preset})
     * and different from current Beacon configuration.
     *
     * @param profile the Preset instance
     * @param characteristics the characteristics storing current Beacon configuration
     * @return new parameters Batch instance
     */
    static Batch select(final Preset profile, final RemoteBluetoothDevice.Characteristics characteristics) {
      SDKPreconditions.checkNotNull(profile, "Preset is null");

      final Batch batch = new Batch();

      final UUID proximityUUID = profile.getProximity();
      if (proximityUUID != null && !characteristics.getProximityUUID().equals(proximityUUID)) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.PROXIMITY, proximityUUID);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.PROXIMITY, proximityUUID);
      }

      final long advertisingInterval = profile.getInterval();
      final long currentInterval = characteristics.getAdvertisingInterval();
      if (currentInterval != advertisingInterval) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.INTERVAL, advertisingInterval);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.INTERVAL, advertisingInterval);
      }

      final int txPower = profile.getTxPower();
      if (txPower != characteristics.getPowerLevel()) {
        batch.PROCESS_ARGUMENTS.put(Constants.Devices.TX_POWER, txPower);
        batch.ROLLBACK_ARGUMENTS.put(Constants.Devices.TX_POWER, txPower);
      }

      return batch;
    }
  }
}
