/*
 * Decompiled with CFR 0.152.
 */
package com.kontakt.sdk.android.ble.dfu;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.os.Handler;
import android.os.Looper;
import com.kontakt.sdk.android.ble.connection.GattController;
import com.kontakt.sdk.android.ble.connection.KontaktDeviceServiceStore;
import com.kontakt.sdk.android.ble.connection.WriteListener;
import com.kontakt.sdk.android.ble.dfu.DfuAuthorizationService;
import com.kontakt.sdk.android.ble.dfu.DfuProgress;
import com.kontakt.sdk.android.ble.dfu.FirmwareUpdateListener;
import com.kontakt.sdk.android.ble.dfu.IDfuController;
import com.kontakt.sdk.android.ble.dfu.KDFUCommand;
import com.kontakt.sdk.android.ble.dfu.KDFUResponse;
import com.kontakt.sdk.android.ble.dfu.Transaction;
import com.kontakt.sdk.android.ble.dfu.firmwares.FirmwareFileCallback;
import com.kontakt.sdk.android.ble.dfu.firmwares.IFirmwareFilesManager;
import com.kontakt.sdk.android.ble.exception.CharacteristicAbsentException;
import com.kontakt.sdk.android.ble.exception.KontaktDfuException;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.util.FileUtils;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.util.ArrayUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

@TargetApi(value=18)
public class DfuController
implements IDfuController {
    private static final String NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID = "00002902-0000-1000-8000-00805f9b34fb";
    private static final int READ_STATE_RESPONSE_EXPECTED_LENGTH = 80;
    private static final int FIRMWARE_HEADER_LENGTH = 64;
    private static final int OPERATION_TIMEOUT_MILLIS = 100;
    private static final int SEND_CHUNK_TIMEOUT_MILLIS = 5;
    private static final int COMMAND_IN_PROGRESS = 0;
    private static final int COMMAND_DONE = 1;
    private final Handler handler = new Handler(Looper.getMainLooper());
    private final IFirmwareFilesManager firmwareFilesManager;
    private final DfuAuthorizationService authorizationService;
    private final KontaktDeviceServiceStore serviceStore;
    private final Firmware firmware;
    byte[] firmwareFileBytes;
    byte[] readStateResponseData = new byte[0];
    long initializeTimestamp;
    final GattController gattController;
    FirmwareUpdateListener firmwareUpdateListener;
    private Transaction nextTransaction;

    public static DfuController create(byte[] firmwareFile, Firmware firmware, GattController gattController, KontaktDeviceServiceStore serviceStore, IFirmwareFilesManager firmwareFilesManager, DfuAuthorizationService authorizationService) {
        return new DfuController(firmwareFile, firmware, gattController, serviceStore, firmwareFilesManager, authorizationService);
    }

    DfuController(byte[] fileBytes, Firmware firmware, GattController gattController, KontaktDeviceServiceStore serviceStore, IFirmwareFilesManager firmwareFilesManager, DfuAuthorizationService authorizationService) {
        SDKPreconditions.checkNotNull(firmware, "Firmware is null.");
        SDKPreconditions.checkNotNull(gattController, "GattController is null.");
        SDKPreconditions.checkNotNull(serviceStore, "KontaktDeviceServiceStore is null.");
        SDKPreconditions.checkNotNull(firmwareFilesManager, "FirmwareFilesManager is null.");
        SDKPreconditions.checkNotNull(authorizationService, "DfuAuthorizationService is null.");
        this.firmware = firmware;
        this.firmwareFileBytes = fileBytes;
        this.gattController = gattController;
        this.serviceStore = serviceStore;
        this.firmwareFilesManager = firmwareFilesManager;
        this.authorizationService = authorizationService;
    }

    @Override
    public void setFirmwareUpdateListener(FirmwareUpdateListener listener) {
        SDKPreconditions.checkNotNull(listener, "FirmwareUpdateListener is null");
        this.firmwareUpdateListener = listener;
    }

    @Override
    public void initialize() {
        SDKPreconditions.checkNotNull(this.firmwareUpdateListener, "DFU Listener must be set before starting an update procedure.");
        this.firmwareUpdateListener.onStarted();
        this.initializeTimestamp = System.currentTimeMillis();
        this.reportProgress(DfuProgress.INITIALIZING.getPercent(), DfuProgress.INITIALIZING.getMessage());
        if (this.firmwareFileBytes == null) {
            this.firmwareFilesManager.getFirmwareFile(this.firmware, this.createFirmwareFileCallback());
        } else {
            this.authorize();
        }
    }

    void authorize() {
        this.reportProgress(DfuProgress.AUTHORIZING.getPercent(), DfuProgress.AUTHORIZING.getMessage());
        this.authorizationService.setAuthorizationCallback(this.createAuthorizationCallback());
        this.authorizationService.authorize();
    }

    void enableDFUResponseNotification() throws CharacteristicAbsentException, ServiceAbsentException {
        this.reportProgress(DfuProgress.ENABLING_NOTIFICATIONS.getPercent(), DfuProgress.ENABLING_NOTIFICATIONS.getMessage());
        final BluetoothDeviceCharacteristic responseCharacteristic = this.serviceStore.getDfuResponseCharacteristic();
        if (!this.gattController.setCharacteristicNotification(responseCharacteristic, true)) {
            this.reportError("Failed to enable notification for response characteristic");
            return;
        }
        this.delay(new Runnable(){

            @Override
            public void run() {
                BluetoothGattDescriptor descriptor = responseCharacteristic.getDescriptor(UUID.fromString(DfuController.NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID));
                descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
                if (!DfuController.this.gattController.writeDescriptor(descriptor)) {
                    DfuController.this.reportError("Failed to enable notification descriptor for response characteristic");
                }
            }
        });
    }

    void enableDataNotification() throws CharacteristicAbsentException, ServiceAbsentException {
        final BluetoothDeviceCharacteristic dataCharacteristic = this.serviceStore.getDfuDataCharacteristic();
        if (!this.gattController.setCharacteristicNotification(dataCharacteristic, true)) {
            this.reportError("Failed to enable notification for data characteristic");
            return;
        }
        this.delay(new Runnable(){

            @Override
            public void run() {
                BluetoothGattDescriptor descriptor = dataCharacteristic.getDescriptor(UUID.fromString(DfuController.NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID));
                descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
                if (!DfuController.this.gattController.writeDescriptor(descriptor)) {
                    DfuController.this.reportError("Failed to enable notification descriptor for data characteristic");
                }
            }
        });
    }

    void readState() {
        this.delay(new Runnable(){

            @Override
            public void run() {
                DfuController.this.sendCommand(KDFUCommand.GET_STATE.getCodeAsArray());
            }
        });
    }

    void eraseStoredData() {
        this.delay(new Runnable(){

            @Override
            public void run() {
                DfuController.this.sendCommand(KDFUCommand.ERASE.getCodeAsArray());
            }
        });
    }

    void startNextTransaction() {
        this.sendCommand(new byte[]{KDFUCommand.START_TRANSACTION.getCode(), (byte)(this.nextTransaction.getSize() & 0xFF), (byte)(this.nextTransaction.getSize() >> 8 & 0xFF)});
    }

    void finalizeTransaction() {
        this.delay(new Runnable(){

            @Override
            public void run() {
                DfuController.this.sendCommand(KDFUCommand.FINALIZE_TRANSACTION.getCodeAsArray());
            }
        });
    }

    private void activateFirmware() {
        this.delay(new Runnable(){

            @Override
            public void run() {
                DfuController.this.sendCommand(KDFUCommand.ACTIVATE.getCodeAsArray());
            }
        });
    }

    private void prepareFirstTransaction(int storedBytesCount) {
        this.nextTransaction = this.createTransaction(this.firmwareFileBytes, storedBytesCount);
        this.reportProgress(DfuProgress.UPLOADING_FIRMWARE.getPercent(), DfuProgress.UPLOADING_FIRMWARE.getMessage());
    }

    private void handleReadStateNotification(byte[] value) {
        this.readStateResponseData = ArrayUtils.concat(this.readStateResponseData, value);
        if (this.readStateResponseData.length != 80) {
            return;
        }
        int storedBytesLength = ArrayUtils.byteArrayToInt(Arrays.copyOfRange(this.readStateResponseData, 4, 8));
        byte[] storedFirmwareHeader = Arrays.copyOfRange(this.readStateResponseData, 16, 80);
        byte[] fileFirmwareHeader = Arrays.copyOfRange(this.firmwareFileBytes, 0, 64);
        if (storedBytesLength == 0 || !Arrays.equals(storedFirmwareHeader, fileFirmwareHeader)) {
            this.eraseStoredData();
        } else {
            this.finalizeTransaction();
            Logger.i("Continuing firmware upload. Stored bytes: " + storedBytesLength);
        }
    }

    private void handleEraseNotification(KDFUResponse response, byte value) {
        if (response != KDFUResponse.SUCCESS) {
            this.reportError("Error while erasing firmware data.");
            return;
        }
        if (value == 1) {
            this.prepareFirstTransaction(0);
            this.startNextTransaction();
        }
    }

    private void handleStartTransactionNotification(KDFUResponse response) {
        if (response == KDFUResponse.SUCCESS) {
            Logger.d("Starting firmware transaction...");
            this.delay(new Runnable(){

                @Override
                public void run() {
                    DfuController.this.prepareTransactionChunks();
                }
            });
        }
    }

    private void handleFinalizeTransactionNotification(KDFUResponse response, byte[] storedSize) {
        int storedBytesCount = ArrayUtils.byteArrayToInt(storedSize);
        if (response == KDFUResponse.SUCCESS) {
            this.reportPercentProgress(storedBytesCount);
            Logger.d("Transaction finalized. Total stored bytes: " + storedBytesCount);
            if (storedBytesCount == this.firmwareFileBytes.length) {
                this.activateFirmware();
                return;
            }
        } else {
            Logger.w("DFU transaction execution failed. Recovering...");
        }
        this.nextTransaction = this.createTransaction(this.firmwareFileBytes, storedBytesCount);
        this.delay(new Runnable(){

            @Override
            public void run() {
                DfuController.this.startNextTransaction();
            }
        });
    }

    private void handleActivateNotification(KDFUResponse response, byte additionalInfoByte) {
        switch (response) {
            case SUCCESS: {
                if (0 == additionalInfoByte) {
                    this.reportProgress(DfuProgress.ACTIVATING_FIRMWARE.getPercent(), DfuProgress.ACTIVATING_FIRMWARE.getMessage());
                    break;
                }
                if (1 != additionalInfoByte) break;
                this.reportProgress(DfuProgress.FIRMWARE_ACTIVATED.getPercent(), DfuProgress.FIRMWARE_ACTIVATED.getMessage());
                this.firmwareUpdateListener.onFinished(System.currentTimeMillis() - this.initializeTimestamp);
                break;
            }
            case IMAGE_CHECKSUM_INVALID: {
                this.reportError("Received image checksum was invalid.");
                break;
            }
            case IMAGE_HEADER_INVALID: {
                this.reportError("Received image header has incorrect format.");
                break;
            }
            case IMAGE_SIZE_INVALID: {
                this.reportError("Received image size was invalid.");
            }
        }
    }

    void prepareTransactionChunks() {
        BluetoothDeviceCharacteristic dataCharacteristic;
        try {
            dataCharacteristic = this.serviceStore.getDfuDataCharacteristic();
        }
        catch (CharacteristicAbsentException | ServiceAbsentException e) {
            this.reportError(e.getMessage());
            return;
        }
        List<byte[]> transactionChunks = this.nextTransaction.getChunksToSend();
        this.sendChunk(0, transactionChunks, dataCharacteristic);
    }

    void sendChunk(final int index, final List<byte[]> transactionChunks, final BluetoothDeviceCharacteristic dataCharacteristic) {
        if (index >= transactionChunks.size()) {
            this.finalizeTransaction();
            return;
        }
        dataCharacteristic.setWriteType(1);
        dataCharacteristic.setValue(transactionChunks.get(index));
        Logger.d("Sending firmware chunk: " + index + " | " + dataCharacteristic.getUuid() + " | " + Arrays.toString(dataCharacteristic.getValue()));
        if (!this.gattController.writeCharacteristic(dataCharacteristic)) {
            this.delay(new Runnable(){

                @Override
                public void run() {
                    DfuController.this.sendChunk(index, transactionChunks, dataCharacteristic);
                }
            }, 100);
            Logger.d("Failed to write transaction chunk characteristic. Retrying...");
            return;
        }
        final int nextIndex = index + 1;
        this.delay(new Runnable(){

            @Override
            public void run() {
                DfuController.this.sendChunk(nextIndex, transactionChunks, dataCharacteristic);
            }
        }, 5);
    }

    void sendCommand(byte[] value) {
        BluetoothDeviceCharacteristic characteristic;
        try {
            characteristic = this.serviceStore.getDfuCommandCharacteristic();
            characteristic.setValue(value);
        }
        catch (CharacteristicAbsentException | ServiceAbsentException e) {
            this.reportError(e.getMessage());
            return;
        }
        if (!this.gattController.writeCharacteristic(characteristic, false)) {
            this.reportError("Failed to write to characteristic: " + characteristic.getName());
        } else {
            Logger.d("Sending command: " + characteristic.getUuid() + " | " + Arrays.toString(characteristic.getValue()));
        }
    }

    @Override
    public void onCharacteristicChanged(BluetoothGattCharacteristic characteristic) {
        BluetoothDeviceCharacteristic bleDeviceCharacteristic = new BluetoothDeviceCharacteristic(characteristic);
        KontaktDeviceCharacteristic kontaktDeviceCharacteristic = bleDeviceCharacteristic.getKontaktDeviceCharacteristic();
        if (kontaktDeviceCharacteristic != KontaktDeviceCharacteristic.KDFU_RESPONSE && kontaktDeviceCharacteristic != KontaktDeviceCharacteristic.KDFU_DATA) {
            this.reportError("Received unknown response.");
            return;
        }
        if (kontaktDeviceCharacteristic == KontaktDeviceCharacteristic.KDFU_DATA) {
            this.handleReadStateNotification(characteristic.getValue());
            return;
        }
        byte[] value = characteristic.getValue();
        KDFUCommand command = KDFUCommand.fromCode(value[0]);
        KDFUResponse response = KDFUResponse.fromCode(value[1]);
        if (command == null) {
            this.reportError("Unknown characteristic change command. Code: " + value[0]);
            return;
        }
        if (response == null) {
            this.reportError("Unknown characteristic change response. Code: " + value[1]);
            return;
        }
        switch (command) {
            case ERASE: {
                byte eraseInfoByte = value[value.length - 1];
                this.handleEraseNotification(response, eraseInfoByte);
                break;
            }
            case START_TRANSACTION: {
                this.handleStartTransactionNotification(response);
                break;
            }
            case FINALIZE_TRANSACTION: {
                byte[] storedSize = Arrays.copyOfRange(value, 2, value.length);
                this.handleFinalizeTransactionNotification(response, storedSize);
                break;
            }
            case ACTIVATE: {
                byte activateInfoByte = value[value.length - 1];
                this.handleActivateNotification(response, activateInfoByte);
            }
        }
    }

    @Override
    public void onWriteSuccess(WriteListener.WriteResponse response) {
        this.authorizationService.onAuthorizationCommandWriteSuccess();
    }

    @Override
    public void onWriteFailure(WriteListener.Cause cause) {
        this.reportError(cause.name());
    }

    @Override
    public void onDescriptorWriteSuccess(BluetoothGattDescriptor descriptor) {
        BluetoothDeviceCharacteristic bleDeviceCharacteristic = new BluetoothDeviceCharacteristic(descriptor.getCharacteristic());
        KontaktDeviceCharacteristic kontaktDeviceCharacteristic = bleDeviceCharacteristic.getKontaktDeviceCharacteristic();
        if (kontaktDeviceCharacteristic == KontaktDeviceCharacteristic.KDFU_RESPONSE) {
            try {
                this.enableDataNotification();
            }
            catch (CharacteristicAbsentException | ServiceAbsentException e) {
                this.reportError(e.getMessage());
            }
        } else if (kontaktDeviceCharacteristic == KontaktDeviceCharacteristic.KDFU_DATA) {
            this.reportProgress(DfuProgress.NOTIFICATIONS_ENABLED.getPercent(), DfuProgress.NOTIFICATIONS_ENABLED.getMessage());
            this.readState();
        }
    }

    @Override
    public void onDescriptorWriteFailure(BluetoothGattDescriptor descriptor) {
        this.reportError("Error while writing to descriptor: " + descriptor.getUuid().toString());
    }

    @Override
    public void close() {
        this.handler.removeCallbacksAndMessages(null);
        this.gattController.close();
        this.firmwareFilesManager.close();
        this.firmwareUpdateListener = null;
    }

    private FirmwareFileCallback createFirmwareFileCallback() {
        return new FirmwareFileCallback(){

            @Override
            public void onFileAvailable(File firmwareFile) {
                try {
                    DfuController.this.firmwareFileBytes = FileUtils.toByteArray(new FileInputStream(firmwareFile));
                    DfuController.this.authorize();
                }
                catch (IOException e) {
                    DfuController.this.reportError(e.getMessage());
                }
            }

            @Override
            public void onError(KontaktDfuException exception) {
                DfuController.this.reportError(exception.getMessage());
            }
        };
    }

    private DfuAuthorizationService.AuthorizationCallback createAuthorizationCallback() {
        return new DfuAuthorizationService.AuthorizationCallback(){

            @Override
            public void onAuthorized() {
                try {
                    DfuController.this.reportProgress(DfuProgress.AUTHORIZED.getPercent(), DfuProgress.AUTHORIZED.getMessage());
                    DfuController.this.enableDFUResponseNotification();
                }
                catch (CharacteristicAbsentException | ServiceAbsentException e) {
                    DfuController.this.reportError(e.getMessage());
                }
            }

            @Override
            public void onError(String message) {
                DfuController.this.reportError(message);
            }
        };
    }

    private Transaction createTransaction(byte[] firmwareFileBytes, int storedBytesCount) {
        return new Transaction(firmwareFileBytes, storedBytesCount);
    }

    void delay(Runnable runnable) {
        this.delay(runnable, 100);
    }

    void delay(Runnable runnable, int delay) {
        this.handler.postDelayed(runnable, (long)delay);
    }

    void reportProgress(int percent, String message) {
        if (this.firmwareUpdateListener != null) {
            this.firmwareUpdateListener.onProgress(percent, message);
        }
    }

    void reportPercentProgress(double storedBytes) {
        double uploadPercent = storedBytes / (double)this.firmwareFileBytes.length * 100.0;
        double delta = DfuProgress.FIRMWARE_UPLOADED.getPercent() - DfuProgress.UPLOADING_FIRMWARE.getPercent();
        double normalizedPercent = delta / 100.0 * uploadPercent + (double)DfuProgress.UPLOADING_FIRMWARE.getPercent();
        this.reportProgress((int)normalizedPercent, DfuProgress.UPLOADING_FIRMWARE.getMessage());
    }

    void reportError(String message) {
        if (this.firmwareUpdateListener != null) {
            this.firmwareUpdateListener.onError(new KontaktDfuException(message));
        }
    }
}

