/*
 * Copyright (C) 2007 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.kontakt.sdk.android.common.util;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.channels.Channel;

/**
 * Closeables provide utility method for closing Streams Channels and Closeables. For more information
 *
 * @see <a href="http://code.google.com/p/guava-libraries/">Guava Libraries project.</a>
 */
public final class Closeables {

  private Closeables() {
  }

  /**
   * Closes object implementing closeable by either swallowing or propagating IOException depending on flag parameter.
   *
   * @param closeable the closeable
   * @param swallowIOException the swallow iO exception
   * @throws IOException the iO exception
   */
  public static void close(Closeable closeable, boolean swallowIOException) throws IOException {
    if (closeable == null) {
      return;
    }
    try {
      closeable.close();
    } catch (IOException e) {
      if (!swallowIOException) {
        throw e;
      }
    }
  }

  /**
   * Closes Closeable and swallows IOException.
   *
   * @param closeable the input stream
   */
  public static void closeQuietly(Closeable closeable) {
    try {
      closeable.close();
    } catch (IOException e) {
      //Swallowed
    }
  }

  /**
   * Closes Input Stream and swallows IOException.
   *
   * @param inputStream the input stream
   */
  public static void closeQuietly(InputStream inputStream) {
    try {
      close(inputStream, true);
    } catch (IOException impossible) {
      throw new AssertionError(impossible);
    }
  }

  /**
   * Closes Output Stream and swallows IOException.
   *
   * @param outputStream the output stream
   */
  public static void closeQuietly(OutputStream outputStream) {
    try {
      close(outputStream, true);
    } catch (IOException impossible) {
      throw new AssertionError(impossible);
    }
  }

  /**
   * Closes Channel and either propagates or swallows exception depending on specified boolean flag.
   *
   * @param channel the channel
   * @param swallowIOException the swallow iO exception
   * @throws IOException the iO exception
   */
  public static void close(final Channel channel, final boolean swallowIOException) throws IOException {
    if (channel == null) {
      return;
    }
    try {
      channel.close();
    } catch (IOException e) {
      if (!swallowIOException) {
        throw e;
      }
    }
  }

  /**
   * Closes Channel quietly.
   *
   * @param channel the channel
   */
  public static void closeQuietly(final Channel channel) {
    try {
      close(channel, false);
    } catch (IOException e) {
      throw new AssertionError(e);
    }
  }

  /**
   * Closes Reader quietly and swallows IOException.
   *
   * @param reader the reader
   */
  public static void closeQuietly(Reader reader) {
    try {
      close(reader, true);
    } catch (IOException impossible) {
      throw new AssertionError(impossible);
    }
  }
}
