package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.List;
import java.util.UUID;

/**
 * Model class that represents JSON structure of Kontakt.io trigger.
 * To create new instance of this class, please use {@link Trigger.Builder}.
 */
public class Trigger implements Parcelable {

  public static final Creator<Trigger> CREATOR = new Creator<Trigger>() {
    @Override
    public Trigger createFromParcel(Parcel source) {
      return new Trigger(source);
    }

    @Override
    public Trigger[] newArray(int size) {
      return new Trigger[size];
    }
  };

  private final UUID id;
  private final String name;
  private final List<String> activityId;
  private final TriggerType type;
  private final TriggerContext context;

  public static Builder builder() {
    return new Builder();
  }

  Trigger(Builder builder) {
    this.id = builder.id;
    this.name = builder.name;
    this.type = builder.type;
    this.activityId = builder.activityIds;
    this.context = builder.context;
  }

  protected Trigger(Parcel in) {
    this.id = (UUID) in.readSerializable();
    this.name = in.readString();
    this.activityId = in.createStringArrayList();
    int tmpType = in.readInt();
    this.type = tmpType == -1 ? null : TriggerType.values()[tmpType];
    this.context = in.readParcelable(TriggerContext.class.getClassLoader());
  }

  private Trigger() {
    this(new Builder());
  }

  @Override
  public boolean equals(Object object) {
    if (object == this) {
      return true;
    }
    if (object == null || !(object instanceof Trigger)) {
      return false;
    }
    final Trigger trigger = (Trigger) object;

    return SDKEqualsBuilder.start()
        .equals(id, trigger.id)
        .equals(name, trigger.name)
        .equals(type, trigger.type)
        .equals(context, trigger.context)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(id).append(name).append(type).append(context).build();
  }

  @Override
  public String toString() {
    return type.name() + " [" + context.getTrackingId() + "]";
  }

  public UUID getId() {
    return id;
  }

  public String getName() {
    return name;
  }

  public TriggerType getType() {
    return type;
  }

  public TriggerContext getContext() {
    return context;
  }

  public List<String> getActivityIds() {
    return activityId;
  }

  /**
   * Builder class that is used to build {@link Trigger} instances from values configured by the setters.
   */
  public static class Builder {

    UUID id;
    String name;
    TriggerType type;
    TriggerContext context;
    List<String> activityIds;

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder name(final String name) {
      this.name = name;
      return this;
    }

    public Builder type(final TriggerType type) {
      this.type = type;
      return this;
    }

    public Builder context(final TriggerContext context) {
      this.context = context;
      return this;
    }

    public Builder activityIds(final List<String> activityId) {
      this.activityIds = activityId;
      return this;
    }

    public Trigger build() {
      return new Trigger(this);
    }
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.id);
    dest.writeString(this.name);
    dest.writeStringList(this.activityId);
    dest.writeInt(this.type == null ? -1 : this.type.ordinal());
    dest.writeParcelable(this.context, flags);
  }
}

