package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.google.gson.annotations.SerializedName;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * A container for future shuffles both for Eddystone and iBeacon profiles that is aggregated inside
 * {@link Device} object.
 * <br><br>
 * To create new instance of this class, please use {@link Shuffles.Builder}.
 */
public class Shuffles implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Shuffles> CREATOR = new Creator<Shuffles>() {
    @Override
    public Shuffles createFromParcel(Parcel source) {
      return new Shuffles(source);
    }

    @Override
    public Shuffles[] newArray(int size) {
      return new Shuffles[size];
    }
  };

  @SerializedName("EDDYSTONE") private List<EddystoneUid> eddystoneShuffles;
  @SerializedName("IBEACON") private List<IBeaconId> iBeaconShuffles;

  private Shuffles() {
    this(new Builder());
  }

  public static Builder builder() {
    return new Builder();
  }

  Shuffles(Builder builder) {
    eddystoneShuffles = builder.eddystoneShuffles;
    iBeaconShuffles = builder.iBeaconShuffles;
  }

  protected Shuffles(Parcel in) {
    this.eddystoneShuffles = in.createTypedArrayList(EddystoneUid.CREATOR);
    this.iBeaconShuffles = in.createTypedArrayList(IBeaconId.CREATOR);
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof Shuffles)) {
      return false;
    }

    if (o == this) {
      return true;
    }

    Shuffles other = (Shuffles) o;
    return SDKEqualsBuilder.start().equals(eddystoneShuffles, other.eddystoneShuffles).equals(iBeaconShuffles, other.iBeaconShuffles).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(eddystoneShuffles).append(iBeaconShuffles).build();
  }

  public List<EddystoneUid> getEddystoneShuffles() {
    return eddystoneShuffles;
  }

  public List<IBeaconId> getIBeaconShuffles() {
    return iBeaconShuffles;
  }

  public List<SecureProfileUid> getSecureProfileShuffles() {
    if(eddystoneShuffles != null) {
     List<SecureProfileUid> secureProfileUids = new ArrayList<>(eddystoneShuffles.size());
      for (EddystoneUid eddystoneShuffle : eddystoneShuffles) {
        secureProfileUids.add(SecureProfileUid.fromEddystoneUid(eddystoneShuffle));
      }
      return secureProfileUids;
    }
    return new ArrayList<>();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeTypedList(eddystoneShuffles);
    dest.writeTypedList(iBeaconShuffles);
  }

  public static class Builder {
    List<EddystoneUid> eddystoneShuffles = new ArrayList<>();
    List<IBeaconId> iBeaconShuffles = new ArrayList<>();

    public Builder eddystoneShuffles(final Collection<EddystoneUid> eddystoneShuffles) {
      SDKPreconditions.checkNotNull(eddystoneShuffles, "eddystone IDs cannot be null");
      for (EddystoneUid eddystoneUid : eddystoneShuffles) {
        SDKPreconditions.checkNotNull(eddystoneUid, "eddystone IDs cannot contain null value");
      }
      this.eddystoneShuffles.clear();
      this.eddystoneShuffles.addAll(eddystoneShuffles);
      return this;
    }

    public Builder iBeaconShuffles(final Collection<IBeaconId> iBeaconShuffles) {
      SDKPreconditions.checkNotNull(iBeaconShuffles, "iBeacon IDs cannot be null");
      for (IBeaconId iBeaconId : iBeaconShuffles) {
        SDKPreconditions.checkNotNull(iBeaconId, "iBeacon IDs cannot contain null value");
      }
      this.iBeaconShuffles.clear();
      this.iBeaconShuffles.addAll(iBeaconShuffles);
      return this;
    }

    public Shuffles build() {
      return new Shuffles(this);
    }
  }
}
