package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;

/**
 * This class expresses a command that can be executed on a {@link Device}. Every secure command
 * consists of encrypted String config value and a device's unique ID.
 * <br><br>
 * We can obtain secure commands via {@link IKontaktCloud}.
 */
public class SecureCommand implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<SecureCommand> CREATOR = new Creator<SecureCommand>() {
    @Override
    public SecureCommand createFromParcel(Parcel source) {
      return new SecureCommand(source);
    }

    @Override
    public SecureCommand[] newArray(int size) {
      return new SecureCommand[size];
    }
  };

  private final String config;
  private final String uniqueId;

  /**
   * Constructs new secure command with specified config and device's unique ID.
   *
   * @param config the secure config encrypted value.
   * @param uniqueId the device's unique ID.
   * @return SecureCommand instance
   */
  public static SecureCommand create(String config, String uniqueId) {
    return new SecureCommand(config, uniqueId);
  }

  private SecureCommand(final String config, final String uniqueId) {
    this.config = config;
    this.uniqueId = uniqueId;
  }

  protected SecureCommand(Parcel in) {
    this.config = in.readString();
    this.uniqueId = in.readString();
  }

  public String getConfig() {
    return config;
  }

  public String getUniqueId() {
    return uniqueId;
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }

    if (obj == null || !(obj instanceof SecureCommand)) {
      return false;
    }

    final SecureCommand command = (SecureCommand) obj;

    return SDKEqualsBuilder.start().equals(config, command.config).equals(uniqueId, command.uniqueId).result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(config).append(uniqueId).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeString(this.config);
    dest.writeString(this.uniqueId);
  }
}
