package com.kontakt.sdk.android.common.model;

import android.os.Parcel;
import android.os.Parcelable;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.util.HashCodeBuilder;
import com.kontakt.sdk.android.common.util.SDKEqualsBuilder;
import java.util.List;
import java.util.UUID;

/**
 * Actions are triggered when a smartphone comes into range of a device. Each of them has its own
 * content ({@link ActionContent}) that could be an image, an audio file or a URL. Providing such a
 * information depends on the {@link Proximity} value, strictly linked to a distance from the device.
 * By specifying the proximity we can manage actions triggering process - a smartphone will trigger
 * the appropriate action depending on its distance from the device. There are two types of actions:
 * <ul>
 * <li>BROWSER delivers a URL which can be launched in a browser application,
 * <li>CONTENT contains one of the following CONTENT category types - IMAGE, AUDIO, VIDEO, TEXT.
 * </ul>
 *
 * We can obtain actions via {@link IKontaktCloud}.
 * <br><br>
 * To create new instance of this class, please use {@link Action.Builder}.
 */
public class Action implements Parcelable {

  /**
   * Parcelable CREATOR constant.
   */
  public static final Creator<Action> CREATOR = new Creator<Action>() {
    @Override
    public Action createFromParcel(Parcel source) {
      return new Action(source);
    }

    @Override
    public Action[] newArray(int size) {
      return new Action[size];
    }
  };

  /**
   * Enumeration of action type.
   */
  public enum Type {
    /**
     * Browser action type.
     */
    BROWSER,

    /**
     * Content action type.
     */
    CONTENT
  }

  private UUID id;
  private Type type;
  private Proximity proximity;
  private String url;
  private ActionContent content;
  private List<String> deviceUniqueIds;

  public static Builder builder() {
    return new Builder();
  }

  private Action() {
    this(new Builder());
  }

  Action(Builder builder) {
    this.id = builder.id;
    this.type = builder.actionType;
    this.proximity = builder.proximity;
    this.url = builder.url;
    this.content = builder.content;
    this.deviceUniqueIds = builder.deviceUniqueIds;
  }

  protected Action(Parcel in) {
    this.id = (UUID) in.readSerializable();
    int tmpType = in.readInt();
    this.type = tmpType == -1 ? null : Type.values()[tmpType];
    int tmpProximity = in.readInt();
    this.proximity = tmpProximity == -1 ? null : Proximity.values()[tmpProximity];
    this.url = in.readString();
    this.content = in.readParcelable(ActionContent.class.getClassLoader());
    this.deviceUniqueIds = in.createStringArrayList();
  }

  public UUID getId() {
    return id;
  }

  public Type getType() {
    return type;
  }

  public Proximity getProximity() {
    return proximity;
  }

  /**
   * Returns a URL. If {@code type == Type.CONTENT} then {@code null} value will be returned.
   *
   * @return the URL.
   */
  public String getUrl() {
    if (type == Type.CONTENT) {
      return null;
    }
    return url;
  }

  /**
   * Returns a content file. If {@code type == Type.BROWSER} then {@code null} value will be returned.
   *
   * @return the content file.
   */
  public ActionContent getContent() {
    if (type == Type.BROWSER) {
      return null;
    }
    return content;
  }

  public List<String> getDeviceUniqueIds() {
    return deviceUniqueIds;
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof Action)) {
      return false;
    }
    if (o == this) {
      return true;
    }

    Action other = (Action) o;
    return SDKEqualsBuilder.start()
        .equals(id, other.id)
        .equals(type, other.type)
        .equals(proximity, other.proximity)
        .equals(url, other.url)
        .equals(content, other.content)
        .equals(deviceUniqueIds, other.deviceUniqueIds)
        .result();
  }

  @Override
  public int hashCode() {
    return HashCodeBuilder.init().append(id).append(type).append(proximity).append(url).append(content).append(deviceUniqueIds).build();
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel dest, int flags) {
    dest.writeSerializable(this.id);
    dest.writeInt(this.type == null ? -1 : this.type.ordinal());
    dest.writeInt(this.proximity == null ? -1 : this.proximity.ordinal());
    dest.writeString(this.url);
    dest.writeParcelable(this.content, flags);
    dest.writeStringList(this.deviceUniqueIds);
  }

  /**
   * Builder class that is used to build {@link Action} instances from values configured by the setters.
   */
  public static class Builder {
    UUID id;
    Type actionType;
    Proximity proximity;
    String url;
    ActionContent content;
    List<String> deviceUniqueIds;

    public Builder id(final UUID id) {
      this.id = id;
      return this;
    }

    public Builder actionType(final Type actionType) {
      this.actionType = actionType;
      return this;
    }

    public Builder proximity(final Proximity proximity) {
      this.proximity = proximity;
      return this;
    }

    public Builder url(final String url) {
      this.url = url;
      return this;
    }

    public Builder content(final ActionContent content) {
      this.content = content;
      return this;
    }

    public Builder devices(final List<String> deviceUniqueIds) {
      this.deviceUniqueIds = deviceUniqueIds;
      return this;
    }

    public Action build() {
      return new Action(this);
    }
  }
}
