package com.kontakt.sdk.android.cloud.api.executor.venues;

import android.util.Base64;
import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.VenuesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.VenuesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Venue;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link VenuesApi}. Use this class if you want to update
 * venues through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   // define updated venue object
 *   kontaktCloud.venues().update(ID)
 *      .with(updatedVenue)
 *      .execute();
 *   </code>
 * </pre>
 * Keep in mind that venue must be specified so invocation of {@code with} method is mandatory.
 * Otherwise an exception will be thrown.
 */
public class UpdateVenueRequestExecutor extends RequestExecutor<String> {

  private final VenuesService venuesService;
  private final UUID id;
  private Venue venue;
  private String imageFile;

  /**
   * Constructs request executor initialized with corresponding service class and venue unique ID.
   *
   * @param venuesService the venues API service.
   * @param id venue unique identifier.
   */
  public UpdateVenueRequestExecutor(final VenuesService venuesService, final UUID id) {
    this.venuesService = venuesService;
    this.id = id;
  }

  /**
   * Specifies venue data to update. The method invocation is obligatory while using this request executor.
   *
   * @param venue updated venue data.
   * @return this request executor.
   */
  public UpdateVenueRequestExecutor with(final Venue venue) {
    SDKPreconditions.checkNotNull(venue, "venue cannot be null");
    this.venue = venue;
    return this;
  }

  /**
   * Specifies new image file.
   *
   * @param file the image file.
   * @return this request executor.
   * @throws IOException instance of IOException
   */
  public UpdateVenueRequestExecutor withImageFile(final File file) throws IOException {
    SDKPreconditions.checkNotNull(file, "file is null");
    SDKPreconditions.checkState(file.exists(), "file does not exist");
    final byte[] bytes = ConversionUtils.convert(file);
    this.imageFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * Specifies new image file.
   *
   * @param bytes the image file as a bytes stream.
   * @return this request executor.
   * @throws IOException instance of IOException
   */
  public UpdateVenueRequestExecutor withImageFile(final byte[] bytes) throws IOException {
    SDKPreconditions.checkNotNull(bytes, "bytes cannot be null");
    this.imageFile = Base64.encodeToString(bytes, Base64.DEFAULT);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return venuesService.updateVenue(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(venue != null, "cannot update venue - specify venue data to update");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Venues.VENUE_ID_PARAMETER, id.toString());
    params.put(CloudConstants.Venues.NAME_PARAMETER, venue.getName());
    params.put(CloudConstants.Venues.DESCRIPTION_PARAMETER, venue.getDescription());
    if (imageFile != null) {
      params.put(CloudConstants.Venues.FILE_PARAMETER, imageFile);
    }
    if (venue.getLatitude() != null) {
      params.put(CloudConstants.Common.LAT_PARAMETER, venue.getLatitude());
    }
    if (venue.getLongitude() != null) {
      params.put(CloudConstants.Common.LNG_PARAMETER, venue.getLongitude());
    }
    return params;
  }
}
