package com.kontakt.sdk.android.cloud.api.executor.triggers;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.TriggersApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.TriggersService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.common.model.Trigger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link TriggersApi}. Use this class if you want to update
 * triggers through fluent API in chained fashion. Example of use:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   // define updated trigger object
 *   kontaktCloud.triggers().update(ID)
 *      .with(updatedTrigger)
 *      .execute();
 *   </code>
 * </pre>
 */
public class UpdateTriggerRequestExecutor extends RequestExecutor<String> {

  private final TriggersService triggersService;
  private final UUID id;
  private Trigger trigger;

  /**
   * Constructs request executor initialized with corresponding service class and trigger unique ID.
   *
   * @param triggersService the triggers API service.
   * @param id trigger unique identifier.
   */
  public UpdateTriggerRequestExecutor(final TriggersService triggersService, final UUID id) {
    this.triggersService = triggersService;
    this.id = id;
  }

  /**
   * Specifies trigger data to update. The method invocation is obligatory while using this request executor.
   *
   * @param trigger updated trigger data.
   * @return this request executor.
   */
  public UpdateTriggerRequestExecutor with(final Trigger trigger) {
    SDKPreconditions.checkNotNull(trigger, "Trigger cannot be null");
    this.trigger = trigger;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<String> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<String> prepareCall() {
    return triggersService.update(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(trigger != null, "cannot update trigger - specify trigger to update");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Triggers.TRIGGER_ID_PARAMETER, id.toString());
    if (trigger.getName() != null) {
      params.put(CloudConstants.Triggers.NAME_PARAMETER, trigger.getName());
    }
    if (trigger.getType() != null) {
      params.put(CloudConstants.Triggers.TYPE_PARAMETER, trigger.getType().name());
    }
    if (trigger.getContext() != null) {
      if (trigger.getContext().getTrackingId() != null) {
        params.put(CloudConstants.Triggers.CONTEXT_TRACKING_ID_PARAMETER, trigger.getContext().getTrackingId());
      }
      if (trigger.getContext().getProximity() != null) {
        params.put(CloudConstants.Triggers.CONTEXT_PROXIMITY_PARAMETER, trigger.getContext().getProximity().name());
      }
      if (trigger.getContext().getSourceId() != null) {
        params.put(CloudConstants.Triggers.CONTEXT_SOURCE_ID_PARAMETER, trigger.getContext().getSourceId());
      }
    }
    return params;
  }
}
