package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.Access;
import com.kontakt.sdk.android.common.model.IBeaconId;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to fetch
 * iBeacon devices through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Devices iBeacons = kontaktCloud.devices().iBeacons()
 *      .withIds(iBeaconIDs)
 *      .maxResult(10)
 *      .execute();
 *   </code>
 * </pre>
 */
public class IBeaconsRequestExecutor extends DevicesBaseRequestExecutor {

  private final DevicesService devicesService;

  private String[] iBeaconIDs;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param devicesService the devices API service.
   */
  public IBeaconsRequestExecutor(DevicesService devicesService) {
    this.devicesService = devicesService;
    profile = DeviceProfile.IBEACON;
  }

  /**
   * Specifies iBeacon unique IDs. Use that method if you want to fetch particular iBeacon devices by ID.
   *
   * @param iBeaconIDs iBeacon unique identifiers.
   * @return this request executor.
   */
  public IBeaconsRequestExecutor withIds(final IBeaconId... iBeaconIDs) {
    SDKPreconditions.checkNotNull(iBeaconIDs, "iBeacon IDs cannot be null");
    final int size = iBeaconIDs.length;
    this.iBeaconIDs = new String[size];
    for (int i = 0; i < size; i++) {
      this.iBeaconIDs[i] = iBeaconIDs[i].toString();
    }
    return this;
  }

  /**
   * Specifies iBeacon unique IDs. Use that method if you want to fetch particular iBeacon devices by ID.
   *
   * @param iBeaconIDs iBeacon unique identifiers.
   * @return this request executor.
   */
  public IBeaconsRequestExecutor withIds(final List<IBeaconId> iBeaconIDs) {
    SDKPreconditions.checkNotNull(iBeaconIDs, "iBeacon IDs cannot be null");
    final int size = iBeaconIDs.size();
    this.iBeaconIDs = new String[size];
    for (int i = 0; i < size; i++) {
      IBeaconId iBeaconId = iBeaconIDs.get(i);
      this.iBeaconIDs[i] = iBeaconId.toString();
    }
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Devices> prepareCall() {
    final Call<Devices> call;
    if (eTag != null) {
      call = devicesService.getDevices(params(), eTag);
    } else {
      call = devicesService.getDevices(params());
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor startIndex(final int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor maxResult(final int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor filter(final String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor access(final Access access) {
    super.access(access);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor managerIds(final UUID... managerIds) {
    super.managerIds(managerIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor managerIds(final List<UUID> managerIds) {
    super.managerIds(managerIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IBeaconsRequestExecutor eTag(final String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (iBeaconIDs != null) {
      params.put(CloudConstants.Devices.IBEACON_ID_PARAMETER, StringUtils.join(iBeaconIDs, ","));
    }
    return params;
  }
}
