package com.kontakt.sdk.android.cloud.api.executor.devices;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.DevicesApi;
import com.kontakt.sdk.android.cloud.api.service.DevicesService;
import com.kontakt.sdk.android.cloud.response.paginated.Devices;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.Order;
import com.kontakt.sdk.android.common.model.Access;
import com.kontakt.sdk.android.common.model.DeviceType;
import com.kontakt.sdk.android.common.model.OrderBy;
import com.kontakt.sdk.android.common.profile.DeviceProfile;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link DevicesApi}. Use this class if you want to fetch
 * devices through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Devices devices = kontaktCloud.devices().fetch()
 *      .managerIds(managerId)
 *      .type(DeviceType.BEACON)
 *      .profile(DeviceProfile.IBEACON)
 *      .startIndex(5)
 *      .maxResult(10)
 *      .filter("major&gt;=155")
 *      .execute();
 *   </code>
 * </pre>
 */
public class DevicesRequestExecutor extends DevicesBaseRequestExecutor {

  private final DevicesService devicesService;

  private String[] uniqueIds;
  private String unassignedFor;

  /**
   * Constructs request executor initialized with corresponding service class.
   *
   * @param devicesService the devices API service.
   */
  public DevicesRequestExecutor(final DevicesService devicesService) {
    this.devicesService = devicesService;
  }

  /**
   * Specifies device unique IDs. Use that method if you want to fetch particular devices by ID.
   *
   * @param uniqueIds device unique identifiers.
   * @return this request executor.
   */
  public DevicesRequestExecutor withIds(final String... uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    this.uniqueIds = uniqueIds;
    return this;
  }

  /**
   * Specifies device unique IDs. Use that method if you want to fetch particular devices by ID.
   *
   * @param uniqueIds device unique identifiers.
   * @return this request executor.
   */
  public DevicesRequestExecutor withIds(final List<String> uniqueIds) {
    SDKPreconditions.checkNotNull(uniqueIds, "IDs cannot be null");
    final int size = uniqueIds.size();
    this.uniqueIds = uniqueIds.toArray(new String[size]);
    return this;
  }

  /**
   * Specifies device type.
   *
   * @param type the device type.
   * @return this request executor.
   */
  public DevicesRequestExecutor type(final DeviceType type) {
    SDKPreconditions.checkNotNull(type, "type cannot be null");
    this.type = type;
    return this;
  }

  /**
   * Specifies device profile.
   *
   * @param profile the device profile.
   * @return this request executor.
   */
  public DevicesRequestExecutor profile(final DeviceProfile profile) {
    SDKPreconditions.checkNotNull(profile, "profile cannot be null");
    this.profile = profile;
    return this;
  }

  /**
   * Use that method if you want to fetch devices that are assigned to the given manager
   * and not assigned to any venue.
   *
   * @param managerId the manager unique identifier.
   * @return this request executor.
   */
  public DevicesRequestExecutor unassignedFor(final UUID managerId) {
    SDKPreconditions.checkNotNull(managerId, "manager id cannot be null");
    this.unassignedFor = managerId.toString();
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Devices> prepareCall() {
    final Call<Devices> call;
    if (eTag != null) {
      if (unassignedFor != null) {
        call = devicesService.getUnassignedDevices(unassignedFor, params(), eTag);
      } else {
        call = devicesService.getDevices(params(), eTag);
      }
    } else {
      if (unassignedFor != null) {
        call = devicesService.getUnassignedDevices(unassignedFor, params());
      } else {
        call = devicesService.getDevices(params());
      }
    }
    return call;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor startIndex(final int startIndex) {
    super.startIndex(startIndex);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor maxResult(final int maxResult) {
    super.maxResult(maxResult);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor filter(final String query) {
    super.filter(query);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor access(final Access access) {
    super.access(access);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor managerIds(final UUID... managerIds) {
    super.managerIds(managerIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor managerIds(final List<UUID> managerIds) {
    super.managerIds(managerIds);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor orderBy(OrderBy orderBy, Order order) {
    super.orderBy(orderBy, order);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DevicesRequestExecutor eTag(final String eTag) {
    super.eTag(eTag);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = super.params();
    if (uniqueIds != null && uniqueIds.length > 0) {
      params.put(CloudConstants.Common.UNIQUE_ID_PARAMETER, StringUtils.join(uniqueIds, ","));
    }
    return params;
  }
}
