package com.kontakt.sdk.android.cloud.api.executor.activities;

import com.kontakt.sdk.android.cloud.CloudConstants;
import com.kontakt.sdk.android.cloud.api.ActivitiesApi;
import com.kontakt.sdk.android.cloud.api.executor.RequestExecutor;
import com.kontakt.sdk.android.cloud.api.service.ActivitiesService;
import com.kontakt.sdk.android.cloud.exception.KontaktCloudException;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.util.StringUtils;
import com.kontakt.sdk.android.common.model.Activity;
import com.kontakt.sdk.android.common.model.ActivityContext;
import com.kontakt.sdk.android.common.model.HttpMethod;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import retrofit2.Call;

/**
 * Request executor provided by {@link ActivitiesApi}. Use this class if you want to create
 * activity through fluent API in chained fashion, for example:
 * <pre>
 *   <code>
 *   IKontaktCloud kontaktCloud = KontaktCloud.newInstance();
 *   Activity activity = new Activity.Builder()
 *      .name("Entered office")
 *      .build();
 *   kontaktCloud.activities().create(activity).execute();
 *   </code>
 * </pre>
 */
public class CreateActivityRequestExecutor extends RequestExecutor<Activity> {

  private final ActivitiesService activitiesService;
  private final Activity activity;

  /**
   * Constructs request executor initialized with corresponding service class and trigger object.
   *
   * @param activitiesService the activities API service.
   * @param activity the activity entity.
   */
  public CreateActivityRequestExecutor(final ActivitiesService activitiesService, final Activity activity) {
    this.activitiesService = activitiesService;
    this.activity = activity;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Activity execute() throws IOException, KontaktCloudException {
    checkPreconditions();
    return super.execute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void execute(final CloudCallback<Activity> callback) {
    checkPreconditions();
    super.execute(callback);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Call<Activity> prepareCall() {
    return activitiesService.create(params());
  }

  private void checkPreconditions() {
    SDKPreconditions.checkState(activity.getName() != null, "Cannot create activity - specify name");
    SDKPreconditions.checkState(activity.getType() != null, "Cannot create activity - specify type");
    ActivityContext context = activity.getContext();
    SDKPreconditions.checkState(context != null, "Cannot create activity - specify context");
    SDKPreconditions.checkState(context.getHttpMethod() != null, "Cannot create activity - specify HTTP method");
    SDKPreconditions.checkState(context.getUrl() != null, "Cannot create activity - specify url");
    if (context.getHttpMethod() == HttpMethod.POST || context.getHttpMethod() == HttpMethod.PUT) {
      SDKPreconditions.checkState(context.getHeaders() != null, "Cannot create activity - specify content-type header for this HTTP method");
      SDKPreconditions.checkState(context.getHeaders().containsKey("content-type"),
          "Cannot create activity - specify content-type header for this HTTP method");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Map<String, String> params() {
    final Map<String, String> params = new HashMap<>();
    params.put(CloudConstants.Activities.NAME_PARAMETER, activity.getName());
    params.put(CloudConstants.Activities.TYPE_PARAMETER, activity.getType().name());
    params.put(CloudConstants.Activities.CONTEXT_URL, activity.getContext().getUrl());
    params.put(CloudConstants.Activities.CONTEXT_HTTP_METHOD, activity.getContext().getHttpMethod().name());

    List<UUID> triggerIds = activity.getTriggerIds();
    if (triggerIds != null && !triggerIds.isEmpty()) {
      params.put(CloudConstants.Activities.TRIGGER_ID_PARAMETER, StringUtils.join(triggerIds, ","));
    }

    Map<String, String> headers = activity.getContext().getHeaders();
    Map<String, String> parameters = activity.getContext().getParameters();
    if (headers != null) {
      for (Map.Entry<String, String> entry : headers.entrySet()) {
        params.put(CloudConstants.Activities.CONTEXT_HEADERS + entry.getKey(), entry.getValue());
      }
    }
    if (parameters != null) {
      for (Map.Entry<String, String> entry : parameters.entrySet()) {
        params.put(CloudConstants.Activities.CONTEXT_PARAMETERS + entry.getKey(), entry.getValue());
      }
    }
    return params;
  }
}
