package com.kontakt.sdk.android.ble.spec;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothGattService;
import android.os.Build;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.UUID;

/**
 * CharacteristicWrapper delegates methods to wrapped BluetoothGattCharacteristic
 * ({@link BluetoothGattCharacteristic}).
 * Please note that CharacteristicWrapper can wrap only characteristics identified by UUID ids
 * described in kontakt.io iBeacon and Eddystone specification.
 *
 * @see <a href="http://docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon Datasheet - version 2.0.</a>
 */
@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class BluetoothDeviceCharacteristic extends BluetoothGattCharacteristic {

  private final String name;
  private final BluetoothGattCharacteristic characteristic;

  private KontaktDeviceCharacteristic kontaktDeviceCharacteristic;

  /**
   * Instantiates a new CharacteristicWrapper.
   *
   * @param characteristic the characteristic
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public BluetoothDeviceCharacteristic(final BluetoothGattCharacteristic characteristic) {
    super(characteristic.getUuid(), characteristic.getProperties(), characteristic.getPermissions());

    UUID characteristicId = characteristic.getUuid();

    kontaktDeviceCharacteristic = KontaktDeviceCharacteristic.fromUuid(characteristic.getUuid());
    SDKPreconditions.checkNotNull(kontaktDeviceCharacteristic, "Unknown characteristic with ID: " + characteristicId);

    this.name = kontaktDeviceCharacteristic.toString();
    this.characteristic = characteristic;
  }

  @Override
  public UUID getUuid() {
    return characteristic.getUuid();
  }

  @Override
  public boolean addDescriptor(BluetoothGattDescriptor descriptor) {
    return characteristic.addDescriptor(descriptor);
  }

  @Override
  public BluetoothGattService getService() {
    return characteristic.getService();
  }

  @Override
  public void setWriteType(int writeType) {
    characteristic.setWriteType(writeType);
  }

  @Override
  public List<BluetoothGattDescriptor> getDescriptors() {
    return characteristic.getDescriptors();
  }

  @Override
  public BluetoothGattDescriptor getDescriptor(UUID uuid) {
    return characteristic.getDescriptor(uuid);
  }

  @Override
  public Integer getIntValue(int formatType, int offset) {
    return characteristic.getIntValue(formatType, offset);
  }

  @Override
  public Float getFloatValue(int formatType, int offset) {
    return characteristic.getFloatValue(formatType, offset);
  }

  @Override
  public String getStringValue(int offset) {
    return characteristic.getStringValue(offset);
  }

  @Override
  public boolean setValue(int value, int formatType, int offset) {
    return characteristic.setValue(value, formatType, offset);
  }

  @Override
  public boolean setValue(int mantissa, int exponent, int formatType, int offset) {
    return characteristic.setValue(mantissa, exponent, formatType, offset);
  }

  /**
   * Gets id of the characteristic. The Ids are described in kontakt.io Beacon
   * specification - version 2.
   *
   * @return the id
   * @see <a href="docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon specification - version 2.</a>
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public UUID getId() {
    return characteristic.getUuid();
  }

  /**
   * Gets characteristic name. The names are described in in kontakt.io Beacon
   * specification - version 2.
   *
   * @return the name
   * @see <a href="docs.kontakt.io/beacon/kontakt-beacon-v2.pdf" target="_blank">kontakt.io Beacon specification - version 2</a>
   */
  public String getName() {
    return name;
  }

  /**
   * Checks whether wrapped characteristic is writable.
   *
   * @return the boolean value specifying whether wrapped characteristic is writable
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public boolean isWritable() {
    return (getProperties() & (BluetoothGattCharacteristic.PROPERTY_WRITE | BluetoothGattCharacteristic.PROPERTY_WRITE_NO_RESPONSE)) != 0;
  }

  /**
   * Checks whether wrapped characteristic is readable.
   *
   * @return the boolean value specifying whether wrapped characteristic is
   * readable
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public boolean isReadable() {
    return (getProperties() & BluetoothGattCharacteristic.PROPERTY_READ) != 0;
  }

  /**
   * Checks whether wrapped characteristic is notifiable.
   *
   * @return the boolean value specifying whether wrapped characteristic is
   * notifiable
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public boolean isNotifiable() {
    return (getProperties() & BluetoothGattCharacteristic.PROPERTY_NOTIFY) != 0;
  }

  /**
   * Sets Integer as characteristic's value.
   *
   * @param value the int value
   */
  public void setValue(final int value) {
    setValue(ByteBuffer.allocate(4).putInt(value).array());
  }

  /**
   * Sets String as characteristic's value.
   *
   * @param value the String value
   */
  @Override
  public boolean setValue(final String value) {
    return setValue(value.getBytes());
  }

  /**
   * Sets Long as characteristic's value.
   *
   * @param value the long value
   * @return set value
   */
  public boolean setValue(final long value) {
    return setValue(ByteBuffer.allocate(4).putLong(value).array());
  }

  @Override
  /**
   * Sets characteristic's value as byte array.
   *
   * @param value the value
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public boolean setValue(final byte[] value) {
    return characteristic.setValue(value);
  }

  /**
   * Gets characteristic's value as byte array.
   *
   * @return the byte array value
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public byte[] getValue() {
    return characteristic.getValue();
  }

  /**
   * Gets characteristic's value as String.
   *
   * @return the byte array value converted to String. If the value is Null, then
   * empty string is returned.
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public String getStringValue() {
    final byte[] value = characteristic.getValue();
    return value != null ? new String(value) : "";
  }

  /**
   * Gets hex string value.
   *
   * @return the hex string value
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public String getHexStringValue() {
    final byte[] value = characteristic.getValue();
    StringBuilder hexStringBuilder = new StringBuilder();
    for (int i = 0; i < value.length; i++) {
      hexStringBuilder.append(Integer.toHexString(value[i] & 0x00FF));
    }
    return hexStringBuilder.toString();
  }

  /**
   * Gets characteristic's value as UUID.
   *
   * @return the byte array value converted to UUID.
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public UUID getUUIDValue() {
    return ConversionUtils.toUUID(characteristic.getValue());
  }

  /**
   * Gets characteristic's value as Integer.
   *
   * @return the byte array value converted to Integer
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public int getIntValue() {
    return ConversionUtils.asInt(characteristic.getValue());
  }

  /**
   * Gets characteristic's valueas Lon.
   *
   * @return the byte array value converted to long
   */
  public long getLongValue() {
    final Integer value = getIntValue();
    return value.longValue();
  }

  /**
   * Gets characteristic permissions.
   *
   * @return the permissions
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public int getPermissions() {
    return characteristic.getPermissions();
  }

  /**
   * Gets characteristic write type.
   *
   * @return the write type
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public int getWriteType() {
    return characteristic.getWriteType();
  }

  /**
   * Gets characteristic properties.
   *
   * @return the properties
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public int getProperties() {
    return characteristic.getProperties();
  }

  /**
   * Gets characteristic instance id.
   *
   * @return the instance id
   */
  @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  public int getInstanceId() {
    return characteristic.getInstanceId();
  }

  public KontaktDeviceCharacteristic getKontaktDeviceCharacteristic() {
    return kontaktDeviceCharacteristic;
  }
}
