package com.kontakt.sdk.android.ble.service;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanRecord;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.os.Build;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.util.SDKPreconditions;
import java.io.IOException;
import java.util.Collection;

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
abstract class MonitorCallbackL extends ScanCallback implements BleScanCallback {

  protected final MonitorCallback wrappedScanCallback;

  MonitorCallbackL(MonitorCallback scanCallback) {
    SDKPreconditions.checkNotNull(scanCallback, "Wrapped scan callback is null");
    this.wrappedScanCallback = scanCallback;
  }

  @Override
  public void onScanResult(int callbackType, ScanResult result) {
    if (callbackType != ScanSettings.CALLBACK_TYPE_ALL_MATCHES) {
      return;
    }

    final ScanRecord scanRecord = result.getScanRecord();

    if (scanRecord == null) {
      return;
    }

    onLeScan(result.getDevice(), result.getRssi(), scanRecord.getBytes());
  }

  @Override
  public void onLeScan(BluetoothDevice device, int rssi, byte[] scanRecord) {
    wrappedScanCallback.onLeScan(device, rssi, scanRecord);
  }

  @Override
  public void onScanFailed(int errorCode) {
    switch (errorCode) {
      case SCAN_FAILED_FEATURE_UNSUPPORTED:
        Logger.d("Scan failed - feature unsupported");
        break;
      case SCAN_FAILED_INTERNAL_ERROR:
        Logger.d("Scan failed - internal error");
        break;
      case SCAN_FAILED_APPLICATION_REGISTRATION_FAILED:
        Logger.d("Scan failed - application registration failed");
        break;
      case SCAN_FAILED_ALREADY_STARTED:
        Logger.d("Scan failed - application already started");
        break;
      default:
        Logger.d("Scan failed - error code = " + errorCode);
    }
  }

  @Override
  public Collection<InternalProximityListener> getMonitoringListeners() {
    return wrappedScanCallback.getMonitoringListeners();
  }

  @Override
  public void addListener(InternalProximityListener proximityListener) {
    wrappedScanCallback.addListener(proximityListener);
  }

  @Override
  public void removeListener(InternalProximityListener proximityListener) {
    wrappedScanCallback.removeListener(proximityListener);
  }

  @Override
  public void close() throws IOException {
    wrappedScanCallback.close();
  }

  void onMonitorStarted() {
    wrappedScanCallback.onMonitorCycleStart();
  }

  void onMonitorStopped() {
    wrappedScanCallback.onMonitorCycleStop();
  }
}
