package com.kontakt.sdk.android.ble.service;

import android.bluetooth.BluetoothDevice;
import android.os.Handler;
import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.ScanPeriod;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.manager.listeners.InternalProximityListener;
import com.kontakt.sdk.android.common.KontaktSDK;
import com.kontakt.sdk.android.common.log.Logger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@SuppressWarnings("MissingPermission")
abstract class MonitorCallback implements BleScanCallback, DiscoveryContract {

  final BluetoothDeviceDiscoverer bluetoothDeviceDiscoverer;
  private final Handler handler = new Handler();
  private final Map<Integer, InternalProximityListener> monitoringListenerMap = new ConcurrentHashMap<>();
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final ScanPeriod scanPeriod;

  MonitorCallback(ScanContext scanContext, Collection<InternalProximityListener> listeners) {
    bluetoothDeviceDiscoverer = new DefaultBluetoothDeviceDiscoverer(scanContext, this);
    activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    scanPeriod = scanContext.getScanPeriod();
    addRegisteredListeners(listeners);
  }

  @Override
  public void onLeScan(final BluetoothDevice device, final int rssi, final byte[] scanRecord) {
    if (!isAnyListenerRegistered() && KontaktSDK.isInitialized()) {
      return;
    }

    if (device == null) {
      Logger.d("Remote device discovered but is null");
      return;
    }
    performDiscovery(device, rssi, scanRecord);
  }

  private void performDiscovery(final BluetoothDevice device, final int rssi, final byte[] scanRecord) {
    handler.post(new Runnable() {
      @Override
      public void run() {
        bluetoothDeviceDiscoverer.performDiscovery(device, rssi, scanRecord);
        Logger.d(String.format("Device discovered %s", device.toString()));
      }
    });
  }

  @Override
  public Collection<InternalProximityListener> getMonitoringListeners() {
    return Collections.unmodifiableCollection(new ArrayList<>(monitoringListenerMap.values()));
  }

  @Override
  public void addListener(InternalProximityListener proximityListener) {
    final int hashCode = System.identityHashCode(proximityListener);
    final InternalProximityListener insertion = monitoringListenerMap.put(hashCode, proximityListener);

    if (insertion == null) {
      Logger.d("MonitoringListener registered: " + hashCode);
    }
  }

  @Override
  public void removeListener(InternalProximityListener proximityListener) {
    final int hashCode = System.identityHashCode(proximityListener);
    final InternalProximityListener deletion = monitoringListenerMap.remove(hashCode);

    if (deletion != null) {
      Logger.d("MonitoringListener unregistered: " + hashCode);
    }
  }

  @Override
  public void onMonitorCycleStart() {
    if (scanPeriod != ScanPeriod.RANGING) {
      for (final InternalProximityListener proximityListener : monitoringListenerMap.values()) {
        handler.post(new Runnable() {
          @Override
          public void run() {
            proximityListener.onMonitoringCycleStart();
          }
        });
      }
    }
    handler.post(new Runnable() {
      @Override
      public void run() {
        startPeriodicInactivityCheck();
      }
    });
  }

  @Override
  public void onMonitorCycleStop() {
    if (scanPeriod != ScanPeriod.RANGING) {
      for (final InternalProximityListener proximityListener : monitoringListenerMap.values()) {
        handler.post(new Runnable() {
          @Override
          public void run() {
            proximityListener.onMonitoringCycleStop();
          }
        });
      }
    }
    handler.post(new Runnable() {
      @Override
      public void run() {
        stopPeriodicInactivityCheck();
      }
    });
  }

  @Override
  public void onEvent(BluetoothDeviceEvent event) {
    for (final InternalProximityListener proximityListener : monitoringListenerMap.values()) {
      proximityListener.onEvent(event);
    }
  }

  void startPeriodicInactivityCheck() {
    if (activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
      return;
    }

    final long checkPeriod = activityCheckConfiguration.getCheckPeriod();
    handler.postDelayed(periodicActivityCheckRunnable, checkPeriod);
  }

  void stopPeriodicInactivityCheck() {
    if (activityCheckConfiguration == ActivityCheckConfiguration.DISABLED) {
      return;
    }
    handler.removeCallbacksAndMessages(null);
  }

  void evictInactiveDevices(long currentTimeMillis) {
    bluetoothDeviceDiscoverer.evictInactiveDevices(currentTimeMillis);
    startPeriodicInactivityCheck();
  }

  private void addRegisteredListeners(Collection<InternalProximityListener> listeners) {
    for (InternalProximityListener proximityListener : listeners) {
      addListener(proximityListener);
    }
  }

  private boolean isAnyListenerRegistered() {
    return !monitoringListenerMap.isEmpty();
  }

  @Override
  public void close() {
    bluetoothDeviceDiscoverer.disable();
    monitoringListenerMap.clear();
    handler.removeCallbacksAndMessages(null);
  }

  private final Runnable periodicActivityCheckRunnable = new Runnable() {
    @Override
    public void run() {
      evictInactiveDevices(System.currentTimeMillis());
    }
  };
}
