package com.kontakt.sdk.android.ble.manager.internal;

import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.exception.ScanError;
import com.kontakt.sdk.android.ble.util.ScanContextUtils;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import com.kontakt.sdk.android.common.profile.IEddystoneNamespace;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

class ShuffledSpacesManager {

  interface OnSpacesResolvedListener {
    void onSpacesResolved(ScanContext scanContext);

    void onError(ScanError error);

    OnSpacesResolvedListener NOOP_LISTENER = new OnSpacesResolvedListener() {
      @Override
      public void onSpacesResolved(ScanContext scanContext) {
        //No operation
      }

      @Override
      public void onError(ScanError error) {
        //No operation
      }
    };
  }

  OnSpacesResolvedListener onSpacesResolvedListener = OnSpacesResolvedListener.NOOP_LISTENER;
  final List<SpacesResolver> resolvers = new ArrayList<>();
  private ScanContext scanContext;

  public ShuffledSpacesManager(IKontaktCloud cloud) {
    checkNotNull(cloud);
    resolvers.add(new SpacesResolver.RegionsResolver(cloud));
    resolvers.add(new SpacesResolver.NamespacesResolver(cloud));
  }

  //Used in tests
  ShuffledSpacesManager(IKontaktCloud cloud, SpacesResolver... resolvers) {
    checkNotNull(cloud);
    Collections.addAll(this.resolvers, resolvers);
  }

  public void resolve(ScanContext scanContext, OnSpacesResolvedListener onSpacesResolvedListener) {
    this.scanContext = checkNotNull(scanContext);
    this.onSpacesResolvedListener = checkNotNull(onSpacesResolvedListener);
    for (SpacesResolver resolver : resolvers) {
      resolver.resolve(scanContext, resolvingStatusListener);
    }
  }

  public void clearCache() {
    for (SpacesResolver resolver : resolvers) {
      resolver.clearCache();
    }
  }

  public void onDestroy() {
    onSpacesResolvedListener = OnSpacesResolvedListener.NOOP_LISTENER;
  }

  ScanContext createScanContext(Collection<IBeaconRegion> resolvedRegions, Collection<IEddystoneNamespace> resolvedNamespaces) {
    return ScanContextUtils.cloneWithNewSpaces(scanContext, resolvedRegions, resolvedNamespaces);
  }

  void resetFinishedStatus() {
    for (SpacesResolver resolver : resolvers) {
      resolver.resetFinishedStatus();
    }
  }

  boolean allResolversAreFinishedSuccessfully() {
    for (SpacesResolver resolver : resolvers) {
      if (!resolver.isFinishedSuccessfully()) {
        return false;
      }
    }
    return true;
  }

  private final SpacesResolver.ResolvingStatusListener resolvingStatusListener = new SpacesResolver.ResolvingStatusListener() {
    @Override
    public void onSuccess() {
      if (allResolversAreFinishedSuccessfully()) {
        Collection<IBeaconRegion> regions = resolvers.get(0).getResolvedSpaces();
        Collection<IEddystoneNamespace> namespaces = resolvers.get(1).getResolvedSpaces();
        resetFinishedStatus();
        onSpacesResolvedListener.onSpacesResolved(createScanContext(regions, namespaces));
      }
    }

    @Override
    public void onError(String message) {
      onSpacesResolvedListener.onError(new ScanError(message));
    }
  };
}
