package com.kontakt.sdk.android.ble.manager;

import com.kontakt.sdk.android.ble.connection.OnServiceReadyListener;
import com.kontakt.sdk.android.ble.manager.configuration.FiltersConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.GeneralConfigurator;
import com.kontakt.sdk.android.ble.manager.configuration.SpacesConfigurator;
import com.kontakt.sdk.android.ble.manager.listeners.EddystoneListener;
import com.kontakt.sdk.android.ble.manager.listeners.IBeaconListener;
import com.kontakt.sdk.android.ble.manager.listeners.ScanStatusListener;
import com.kontakt.sdk.android.ble.manager.listeners.SecureProfileListener;
import com.kontakt.sdk.android.ble.manager.listeners.SpaceListener;
import com.kontakt.sdk.android.ble.service.ProximityService;

public interface ProximityManagerContract {

  /**
   * Connects to backing {@link ProximityService}. This needs to be done before first calling {@link ProximityManagerContract#startScanning()}.
   *
   * @param onServiceReadyListener Listener used to specify if ProximityManager has successfully connected to backing service. This means that all
   * necessary Android permissions have been granted and ProximityManager is ready to start scanning.
   */
  void connect(OnServiceReadyListener onServiceReadyListener);

  /**
   * Finishes scanning and disconnects from backing service. Calling {@link ProximityManagerContract#connect(OnServiceReadyListener)} is required if
   * scanning should be started again.
   */
  void disconnect();

  /**
   * Starts scanning for Kontakt.io devices.
   */
  void startScanning();

  /**
   * Stops scanning. Does not disconnect from backing service. Call {@link ProximityManagerContract#disconnect()} to disconnect.
   */
  void stopScanning();

  /**
   * Stops scanning, applies latest configuration and restarts scanning.
   */
  void restartScanning();

  /**
   * @return True if ProximityManager is connected to backing service and ready to scan.
   */
  boolean isConnected();

  /**
   * @return True if scanning is currently in progress.
   */
  boolean isScanning();

  /**
   * Sets {@link ScanStatusListener}.
   *
   * @param listener {@link ScanStatusListener}. Nullable.
   */
  void setScanStatusListener(ScanStatusListener listener);

  /**
   * Sets {@link SpaceListener}.
   *
   * @param listener {@link SpaceListener}. Nullable.
   */
  void setSpaceListener(SpaceListener listener);

  /**
   * Sets {@link IBeaconListener}.
   *
   * @param listener {@link IBeaconListener}. Nullable.
   */
  void setIBeaconListener(IBeaconListener listener);

  /**
   * Sets {@link EddystoneListener}.
   *
   * @param listener {@link EddystoneListener}. Nullable.
   */
  void setEddystoneListener(EddystoneListener listener);

  /**
   * Sets {@link SecureProfileListener}. This should be used only when working with Kontakt.io Beacon PRO devices.
   *
   * @param listener {@link SecureProfileListener}. Nullable.
   */
  void setKontaktSecureProfileListener(SecureProfileListener listener);

  /**
   * Configure scanning related options.
   *
   * @return Configuration
   */
  GeneralConfigurator configuration();

  /**
   * Configure IBeacon regions and Eddystone namespaces.
   *
   * @return ProximityManagerSpaces
   */
  SpacesConfigurator spaces();

  /**
   * Configure filtering.
   *
   * @return ProximityManagerFilters
   */
  FiltersConfigurator filters();
}
