package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.bluetooth.BluetoothDevice;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.device.SecureProfile;
import com.kontakt.sdk.android.ble.discovery.FrameDataType;
import com.kontakt.sdk.android.ble.discovery.Parser;
import com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver;
import com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import com.kontakt.sdk.android.common.util.ConversionUtils;
import java.util.Arrays;

import static com.kontakt.sdk.android.ble.discovery.eddystone.InstanceIdResolver.SECURE_PROFILE_INSTANCE_ID_START_INDEX;
import static com.kontakt.sdk.android.ble.discovery.eddystone.NamespaceIdResolver.SECURE_PROFILE_NAMESPACE_ID_START_INDEX;

@SuppressWarnings("MissingPermission")
public final class SecureProfileParser extends Parser<SecureProfile> {

  private static final byte[] KONTAKT_SECURE_PROFILE_PREFIX = new byte[] { 0x16, 0x6A, (byte) 0xFE };
  private static final int MIN_SECURE_PROFILE_LENGTH = 14;
  private static final byte PLAIN_DEVICE_IDENTIFIER_PAYLOAD = 0x02;
  private static final byte SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD = 0x01;
  private static final NamespaceIdResolver NAMESPACE_RESOLVER = new NamespaceIdResolver(SECURE_PROFILE_NAMESPACE_ID_START_INDEX);
  private static final InstanceIdResolver INSTANCE_ID_RESOLVER = new InstanceIdResolver(SECURE_PROFILE_INSTANCE_ID_START_INDEX);

  SecureProfileParser(final ScanContext scanContext) {
    super(scanContext);
  }

  boolean isValidSecureProfileFrame(final byte[] scanRecord) {
    return ConversionUtils.doesArrayContainSubset(scanRecord, KONTAKT_SECURE_PROFILE_PREFIX, 4) && scanRecord.length > MIN_SECURE_PROFILE_LENGTH;
  }

  void parseScanRecord(final byte[] scanRecord) {
    frameData.clear();
    extractFrameData(scanRecord, frameData);
  }

  ISecureProfile getSecureProfile(final BluetoothDevice device, int rssi) {

    int deviceHashCode = hashCodeBuilder
        .append(device.getAddress())
        .append(frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA))
        .build();

    SecureProfile secureDevice = devicesCache.get(deviceHashCode);
    if (secureDevice != null) {
      update(secureDevice, rssi);
      return secureDevice;
    }

    byte[] serviceData = frameData.get(FrameDataType.SCAN_RESPONSE_SERVICE_DATA);
    byte[] nameData = frameData.get(FrameDataType.LOCAL_NAME);

    byte payload = serviceData[2];
    int batteryLevel = serviceData[6];
    int txPower = serviceData[7];
    boolean isShuffled = (payload == SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD);
    String name = parseName(nameData);
    String uniqueId = parseUniqueId(serviceData, payload);
    String namespace = parseNamespace(serviceData, payload);
    String shuffledInstanceId = parseShuffledInstanceId(serviceData, payload);

    secureDevice = new SecureProfile.Builder().macAddress(device.getAddress())
        .name(name)
        .uniqueId(uniqueId)
        .batteryLevel(batteryLevel)
        .firmwareRevision(parseFirmwareVersion(serviceData))
        .txPower(txPower)
        .namespace(namespace)
        .instanceId(shuffledInstanceId)
        .rssi(rssi)
        .shuffled(isShuffled)
        .build();

    devicesCache.put(deviceHashCode, secureDevice);

    return secureDevice;
  }

  private void update(SecureProfile device, int rssi) {
    int calculatedRssi = rssiCalculator.calculateRssi(device.getMacAddress().hashCode(), rssi);
    device.setRssi(calculatedRssi);
  }

  private String parseName(final byte[] nameData) {
    if (nameData == null) {
      return null;
    }
    return new String(nameData);
  }

  private String parseFirmwareVersion(final byte[] serviceData) {
    final int firmwareVersionMajor = Integer.parseInt(String.valueOf(serviceData[4]), 16);
    final int firmwareVersionMinor = Integer.parseInt(String.valueOf(serviceData[5]), 16);
    return String.format("%d.%d", firmwareVersionMajor, firmwareVersionMinor);
  }

  private String parseUniqueId(final byte[] serviceData, int payloadId) {
    if (payloadId == PLAIN_DEVICE_IDENTIFIER_PAYLOAD) {
      return new String(Arrays.copyOfRange(serviceData, 8, serviceData.length));
    }
    return null;
  }

  private String parseNamespace(final byte[] serviceData, int payloadId) {
    if (payloadId == SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD) {
      return NAMESPACE_RESOLVER.parse(serviceData);
    }
    return null;
  }

  private String parseShuffledInstanceId(final byte[] serviceData, int payloadId) {
    if (payloadId == SHUFFLED_DEVICE_IDENTIFIER_PAYLOAD) {
      return INSTANCE_ID_RESOLVER.parse(serviceData);
    }
    return null;
  }

  @Override
  protected void disable() {
    if (isEnabled) {
      isEnabled = false;
    }
  }
}
