package com.kontakt.sdk.android.ble.discovery.secure_profile;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.os.Build;
import android.util.SparseLongArray;
import com.kontakt.sdk.android.ble.configuration.ActivityCheckConfiguration;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.ble.util.SafeSparseLongArray;
import com.kontakt.sdk.android.common.profile.ISecureProfile;
import java.util.ArrayList;
import java.util.List;

import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewDiscovered;
import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewLost;
import static com.kontakt.sdk.android.ble.discovery.secure_profile.SecureProfileEvent.createNewUpdated;
import static java.lang.System.currentTimeMillis;

@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class SecureProfileDiscoverer implements BluetoothDeviceDiscoverer {

  private final List<ISecureProfile> cache;
  private final SecureProfileParser parser;
  private final ActivityCheckConfiguration activityCheckConfiguration;
  private final DiscoveryContract discoveryContract;
  private final SparseLongArray secureProfileTimestampArray;
  private final long deviceUpdateIntervalCallback;
  private long lastUpdateCallbackMillis;

  public SecureProfileDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract) {
    this.deviceUpdateIntervalCallback = scanContext.getDevicesUpdateCallbackInterval();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.discoveryContract = discoveryContract;
    this.secureProfileTimestampArray = new SafeSparseLongArray();
    this.parser = new SecureProfileParser(scanContext);
    this.cache = new ReplacingArrayList<>();
  }

  // used for tests
  SecureProfileDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract, SparseLongArray secureProfileTimestampArray,
      List<ISecureProfile> cache) {
    this.deviceUpdateIntervalCallback = scanContext.getDevicesUpdateCallbackInterval();
    this.activityCheckConfiguration = scanContext.getActivityCheckConfiguration();
    this.discoveryContract = discoveryContract;
    this.secureProfileTimestampArray = secureProfileTimestampArray;
    this.parser = new SecureProfileParser(scanContext);
    this.cache = cache;
  }

  @Override
  public boolean performDiscovery(BluetoothDevice bluetoothDevice, int rssi, byte[] scanResult) {
    if (!parser.isValidSecureProfileFrame(scanResult)) {
      return false;
    }

    parser.parseScanRecord(scanResult);
    ISecureProfile secureProfile = parser.getSecureProfile(bluetoothDevice, rssi);
    notifySecureProfilePresent(secureProfile);

    boolean isDiscovered = cache.contains(secureProfile);
    if (!isDiscovered) {
      cache.add(secureProfile);
      onDiscovered(secureProfile);
    } else {
      onUpdated();
    }
    return true;
  }

  @Override
  public void disable() {
    parser.disable();
  }

  @Override
  public void evictInactiveDevices(long currentTimeMillis) {
    final long inactivityTimeout = activityCheckConfiguration.getInactivityTimeout();

    for (ISecureProfile secureProfile : new ArrayList<>(cache)) {
      long timestamp = getSecureProfileTimestamp(secureProfile);
      if (timestamp != -1L) {
        long threshold = currentTimeMillis - timestamp;
        if (threshold > inactivityTimeout) {
          cache.remove(secureProfile);
          clearBeforeLost(secureProfile);
          onLost(secureProfile);
          removeSecureProfileTimestamp(secureProfile);
        }
      }
    }
  }

  private void notifySecureProfilePresent(ISecureProfile secureProfile) {
    secureProfileTimestampArray.put(secureProfile.getMacAddress().hashCode(), currentTimeMillis());
  }

  private void onDiscovered(ISecureProfile secureProfile) {
    discoveryContract.onEvent(createNewDiscovered(secureProfile, currentTimeMillis()));
  }

  private void onUpdated() {
    long currentTimeMillis = currentTimeMillis();
    if (currentTimeMillis - lastUpdateCallbackMillis > deviceUpdateIntervalCallback) {
      discoveryContract.onEvent(createNewUpdated(cache, currentTimeMillis));
      lastUpdateCallbackMillis = currentTimeMillis;
    }
  }

  private void onLost(ISecureProfile secureProfile) {
    discoveryContract.onEvent(createNewLost(secureProfile, currentTimeMillis()));
  }

  private void clearBeforeLost(ISecureProfile secureProfile) {
    parser.clearRssiCalculation(secureProfile.getMacAddress().hashCode());
  }

  private long getSecureProfileTimestamp(ISecureProfile secureProfile) {
    return secureProfileTimestampArray.get(secureProfile.getMacAddress().hashCode(), -1L);
  }

  private void removeSecureProfileTimestamp(ISecureProfile secureProfile) {
    int index = secureProfileTimestampArray.indexOfKey(secureProfile.getMacAddress().hashCode());
    if (index >= 0) {
      secureProfileTimestampArray.removeAt(index);
    }
  }
}
