package com.kontakt.sdk.android.ble.discovery.ibeacon;

import android.bluetooth.BluetoothDevice;
import com.kontakt.sdk.android.ble.configuration.scan.ScanContext;
import com.kontakt.sdk.android.ble.discovery.AbstractBluetoothDeviceDiscoverer;
import com.kontakt.sdk.android.ble.discovery.BluetoothDeviceEvent;
import com.kontakt.sdk.android.ble.discovery.DiscoveryContract;
import com.kontakt.sdk.android.ble.discovery.EventType;
import com.kontakt.sdk.android.ble.discovery.Validator;
import com.kontakt.sdk.android.ble.util.ReplacingArrayList;
import com.kontakt.sdk.android.common.Proximity;
import com.kontakt.sdk.android.common.profile.IBeaconDevice;
import com.kontakt.sdk.android.common.profile.IBeaconRegion;
import java.util.List;

public class IBeaconDiscoverer extends AbstractBluetoothDeviceDiscoverer<IBeaconRegion, IBeaconDevice> {

  private final IBeaconParser parser;
  private final Validator<IBeaconDevice, IBeaconRegion> validator;

  public IBeaconDiscoverer(ScanContext scanContext, DiscoveryContract discoveryContract) {
    super(discoveryContract, scanContext.getActivityCheckConfiguration(), scanContext.getIBeaconRegions(),
        scanContext.getDevicesUpdateCallbackInterval());

    this.parser = new IBeaconParser(scanContext);
    this.validator = new RegionValidator();
  }

  @Override
  protected BluetoothDeviceEvent createEvent(EventType eventType, IBeaconRegion iBeaconRegion, List<IBeaconDevice> deviceList) {
    return new IBeaconDeviceEvent(eventType, iBeaconRegion, deviceList);
  }

  @Override
  protected void onBeforeDeviceLost(IBeaconDevice device) {
    parser.clearRssiCalculation(device.getAddress().hashCode());
  }

  @Override
  public boolean performDiscovery(BluetoothDevice device, int rssi, byte[] scanRecord) {
    if (!parser.isEnabled() || !parser.isValidIBeaconFrame(scanRecord)) {
      return PROFILE_UNRECOGNIZED;
    }

    parser.parseScanRecord(scanRecord);

    if (parser.getFrameData().size() == 0 || !parser.isManufacturerDataValid()) {
      return PROFILE_UNRECOGNIZED;
    }

    IBeaconDevice iBeaconDevice = parser.getIBeaconDevice(device, rssi);
    notifyDevicePresent(device.getAddress().hashCode(), System.currentTimeMillis());

    final IBeaconRegion iBeaconRegion = extractRegion(iBeaconDevice);
    if (iBeaconRegion == null) {
      return PROFILE_RECOGNIZED_NO_BELONGING_SPACE_FOUND;
    }

    if (!parser.filter(iBeaconDevice)) {
      return PROFILE_RECOGNIZED_FILTERING_NOT_PASSED;
    }

    if (iBeaconDevice.getProximity() != Proximity.UNKNOWN) {
      notifySpacePresent(iBeaconRegion.hashCode(), System.currentTimeMillis());

      ReplacingArrayList<IBeaconDevice> iBeaconList = getDevicesInSpace(iBeaconRegion);
      if (iBeaconList == null) {
        iBeaconList = new ReplacingArrayList<>();
        insertDevicesIntoSpace(iBeaconRegion, iBeaconList);
        onSpaceEnteredEvent(iBeaconRegion);
      }

      if (iBeaconList.addOrReplace(iBeaconDevice)) {
        onDeviceDiscoveredEvent(iBeaconRegion, iBeaconDevice);
      } else {
        onDevicesUpdatedEvent(iBeaconRegion, iBeaconList);
      }

      return true;
    }

    return false;
  }

  @Override
  public void disable() {
    parser.disable();
  }

  private IBeaconRegion extractRegion(final IBeaconDevice iBeaconDevice) {
    for (final IBeaconRegion iBeaconRegion : getSpaceSet()) {
      if (validator.isValid(iBeaconDevice, iBeaconRegion)) {
        return iBeaconRegion;
      }
    }
    return null;
  }
}
