package com.kontakt.sdk.android.ble.dfu.firmwares;

import android.content.Context;
import com.kontakt.sdk.android.ble.exception.KontaktDfuException;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.CloudError;
import com.kontakt.sdk.android.cloud.response.CloudHeaders;
import com.kontakt.sdk.android.common.FileData;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Firmware;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

public class FirmwareFilesManager implements IFirmwareFilesManager {

  private final Context context;
  private final IKontaktCloud kontaktCloud;
  private FirmwareFileCallback callback;

  public static IFirmwareFilesManager create(Context context, IKontaktCloud kontaktCloud) {
    return new FirmwareFilesManager(context, kontaktCloud);
  }

  FirmwareFilesManager(Context context, IKontaktCloud kontaktCloud) {
    checkNotNull(context);
    checkNotNull(kontaktCloud);
    this.context = context.getApplicationContext();
    this.kontaktCloud = kontaktCloud;
  }

  @Override
  public void getFirmwareFile(final Firmware firmware, final FirmwareFileCallback callback) {
    checkNotNull(callback, "FirmwareFileCallback is null");
    checkNotNull(firmware, "Firmware is null");
    checkNotNull(firmware.getFileUrl(), "Firmware's file URL is null");
    checkNotNull(firmware.getId(), "Firmware's file ID is null");

    this.callback = callback;

    Logger.d("Getting firmware file...");
    File file = getCachedFirmwareFile(firmware);
    if (file != null) {
      Logger.d("Firmware file retrieved from disk.");
      callback.onFileAvailable(file);
      return;
    }

    downloadFirmwareFile(firmware, callback);
  }

  private void downloadFirmwareFile(final Firmware firmware, final FirmwareFileCallback callback) {
    Logger.d("Downloading firmware file for " + firmware.getId().toString());
    kontaktCloud.firmwares().fileOfUrl(firmware.getFileUrl()).execute(new CloudCallback<FileData>() {
      @Override
      public void onSuccess(FileData response, CloudHeaders headers) {
        Logger.d("Firmware downloaded.");
        saveFirmware(firmware, response.getData());
        File firmwareFile = getCachedFirmwareFile(firmware);
        if (firmwareFile != null && callback != null) {
          callback.onFileAvailable(firmwareFile);
        } else {
          reportError("Firmware file is missing. Please try again.");
        }
      }

      @Override
      public void onError(CloudError error) {
        reportError(error.getMessage());
      }
    });
  }

  void saveFirmware(Firmware firmware, byte[] firmwareBytes) {
    Logger.d("Saving firmware on disk");
    FileOutputStream fileOutputStream = null;
    try {
      String fileName = firmware.getId().toString();
      fileOutputStream = context.openFileOutput(fileName, Context.MODE_PRIVATE);
      fileOutputStream.write(firmwareBytes);
      Logger.d(String.format("Firmware %s saved on disk", firmware.getId().toString()));
    } catch (IOException e) {
      reportError(e.getMessage());
      Logger.e("Error while saving firmware file on disk: " + e.getMessage());
    } finally {
      if (fileOutputStream != null) {
        try {
          fileOutputStream.close();
        } catch (IOException e) {
          Logger.e("Error while saving firmware file on disk: " + e.getMessage());
          reportError(e.getMessage());
        }
      }
    }
  }

  File getCachedFirmwareFile(Firmware firmware) {
    File filesDir = context.getFilesDir();
    for (File file : filesDir.listFiles()) {
      if (file.getName().contains(firmware.getId().toString()) && file.exists()) {
        return file;
      }
    }
    return null;
  }

  void reportError(String message) {
    if (callback != null) {
      callback.onError(new KontaktDfuException("Error while downloading firmware file: " + message));
    }
  }

  @Override
  public void close() {
    callback = null;
  }
}
