package com.kontakt.sdk.android.ble.dfu;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import com.kontakt.sdk.android.ble.connection.GattController;
import com.kontakt.sdk.android.ble.connection.KontaktDeviceServiceStore;
import com.kontakt.sdk.android.ble.dfu.firmwares.FirmwareFileCallback;
import com.kontakt.sdk.android.ble.dfu.firmwares.IFirmwareFilesManager;
import com.kontakt.sdk.android.ble.exception.CharacteristicAbsentException;
import com.kontakt.sdk.android.ble.exception.KontaktDfuException;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.ble.spec.KontaktDeviceCharacteristic;
import com.kontakt.sdk.android.ble.util.FileUtils;
import com.kontakt.sdk.android.common.log.Logger;
import com.kontakt.sdk.android.common.model.Firmware;
import com.kontakt.sdk.android.common.util.ArrayUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import static com.kontakt.sdk.android.ble.dfu.DfuProgress.ACTIVATING_FIRMWARE;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.AUTHORIZED;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.AUTHORIZING;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.ENABLING_NOTIFICATIONS;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.FIRMWARE_ACTIVATED;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.FIRMWARE_UPLOADED;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.INITIALIZING;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.NOTIFICATIONS_ENABLED;
import static com.kontakt.sdk.android.ble.dfu.DfuProgress.UPLOADING_FIRMWARE;
import static com.kontakt.sdk.android.ble.dfu.KDFUCommand.GET_STATE;
import static com.kontakt.sdk.android.common.util.ArrayUtils.byteArrayToInt;
import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * DfuController class is responsible for updating firmware for Beacon PRO devices.
 */
@TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
public class DfuController implements IDfuController {

  private static final String NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID = "00002902-0000-1000-8000-00805f9b34fb";

  private static final int READ_STATE_RESPONSE_EXPECTED_LENGTH = 80;
  private static final int FIRMWARE_HEADER_LENGTH = 64;

  private static final int OPERATION_TIMEOUT_MILLIS = 100;
  private static final int SEND_CHUNK_TIMEOUT_MILLIS = 5;

  private static final int COMMAND_IN_PROGRESS = 0;
  private static final int COMMAND_DONE = 1;

  private final Handler handler = new Handler(Looper.getMainLooper());
  private final IFirmwareFilesManager firmwareFilesManager;
  private final DfuAuthorizationService authorizationService;
  private final KontaktDeviceServiceStore serviceStore;
  private final Firmware firmware;
  byte[] firmwareFileBytes;
  byte[] readStateResponseData = new byte[0];
  long initializeTimestamp;
  final GattController gattController;
  FirmwareUpdateListener firmwareUpdateListener;
  private Transaction nextTransaction;

  public static DfuController create(byte[] firmwareFile, Firmware firmware, GattController gattController, KontaktDeviceServiceStore serviceStore,
      IFirmwareFilesManager firmwareFilesManager, DfuAuthorizationService authorizationService) {
    return new DfuController(firmwareFile, firmware, gattController, serviceStore, firmwareFilesManager, authorizationService);
  }

  DfuController(byte[] fileBytes, Firmware firmware, GattController gattController, KontaktDeviceServiceStore serviceStore,
      IFirmwareFilesManager firmwareFilesManager, DfuAuthorizationService authorizationService) {
    checkNotNull(firmware, "Firmware is null.");
    checkNotNull(gattController, "GattController is null.");
    checkNotNull(serviceStore, "KontaktDeviceServiceStore is null.");
    checkNotNull(firmwareFilesManager, "FirmwareFilesManager is null.");
    checkNotNull(authorizationService, "DfuAuthorizationService is null.");
    this.firmware = firmware;
    this.firmwareFileBytes = fileBytes;
    this.gattController = gattController;
    this.serviceStore = serviceStore;
    this.firmwareFilesManager = firmwareFilesManager;
    this.authorizationService = authorizationService;
  }

  @Override
  public void setFirmwareUpdateListener(FirmwareUpdateListener listener) {
    checkNotNull(listener, "FirmwareUpdateListener is null");
    firmwareUpdateListener = listener;
  }

  @Override
  public void initialize() {
    checkNotNull(firmwareUpdateListener, "DFU Listener must be set before starting an update procedure.");
    firmwareUpdateListener.onStarted();
    initializeTimestamp = System.currentTimeMillis();
    reportProgress(INITIALIZING.getPercent(), INITIALIZING.getMessage());
    if (firmwareFileBytes == null) {
      firmwareFilesManager.getFirmwareFile(firmware, createFirmwareFileCallback());
    } else {
      authorize();
    }
  }

  void authorize() {
    reportProgress(AUTHORIZING.getPercent(), AUTHORIZING.getMessage());
    authorizationService.setAuthorizationCallback(createAuthorizationCallback());
    authorizationService.authorize();
  }

  void enableDFUResponseNotification() throws CharacteristicAbsentException, ServiceAbsentException {
    reportProgress(ENABLING_NOTIFICATIONS.getPercent(), ENABLING_NOTIFICATIONS.getMessage());

    final BluetoothDeviceCharacteristic responseCharacteristic = serviceStore.getDfuResponseCharacteristic();
    if (!gattController.setCharacteristicNotification(responseCharacteristic, true)) {
      reportError("Failed to enable notification for response characteristic");
      return;
    }

    delay(new Runnable() {
      @Override
      public void run() {
        BluetoothGattDescriptor descriptor = responseCharacteristic.getDescriptor(UUID.fromString(NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID));
        descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
        if (!gattController.writeDescriptor(descriptor)) {
          reportError("Failed to enable notification descriptor for response characteristic");
        }
      }
    });
  }

  void enableDataNotification() throws CharacteristicAbsentException, ServiceAbsentException {
    final BluetoothDeviceCharacteristic dataCharacteristic = serviceStore.getDfuDataCharacteristic();
    if (!gattController.setCharacteristicNotification(dataCharacteristic, true)) {
      reportError("Failed to enable notification for data characteristic");
      return;
    }

    delay(new Runnable() {
      @Override
      public void run() {
        BluetoothGattDescriptor descriptor = dataCharacteristic.getDescriptor(UUID.fromString(NOTIFICATION_CONFIGURATION_DESCRIPTOR_UUID));
        descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE);
        if (!gattController.writeDescriptor(descriptor)) {
          reportError("Failed to enable notification descriptor for data characteristic");
        }
      }
    });
  }

  void readState() {
    delay(new Runnable() {
      @Override
      public void run() {
        sendCommand(GET_STATE.getCodeAsArray());
      }
    });
  }

  void eraseStoredData() {
    delay(new Runnable() {
      @Override
      public void run() {
        sendCommand(KDFUCommand.ERASE.getCodeAsArray());
      }
    });
  }

  void startNextTransaction() {
    //We need to add transaction chunk length (2 bytes little-endian).
    sendCommand(new byte[] {
        KDFUCommand.START_TRANSACTION.getCode(), (byte) (nextTransaction.getSize() & 0xFF), (byte) ((nextTransaction.getSize() >> 8) & 0xFF)
    });
  }

  void finalizeTransaction() {
    delay(new Runnable() {
      @Override
      public void run() {
        sendCommand(KDFUCommand.FINALIZE_TRANSACTION.getCodeAsArray());
      }
    });
  }

  private void activateFirmware() {
    delay(new Runnable() {
      @Override
      public void run() {
        sendCommand(KDFUCommand.ACTIVATE.getCodeAsArray());
      }
    });
  }

  private void prepareFirstTransaction(int storedBytesCount) {
    nextTransaction = createTransaction(firmwareFileBytes, storedBytesCount);
    reportProgress(UPLOADING_FIRMWARE.getPercent(), UPLOADING_FIRMWARE.getMessage());
  }

  private void handleReadStateNotification(byte[] value) {
    readStateResponseData = ArrayUtils.concat(readStateResponseData, value);
    if (readStateResponseData.length != READ_STATE_RESPONSE_EXPECTED_LENGTH) {
      //Wait for rest of the data to be notified.
      return;
    }

    int storedBytesLength = byteArrayToInt(Arrays.copyOfRange(readStateResponseData, 4, 8));
    byte[] storedFirmwareHeader = Arrays.copyOfRange(readStateResponseData, 16, 16 + FIRMWARE_HEADER_LENGTH);
    byte[] fileFirmwareHeader = Arrays.copyOfRange(firmwareFileBytes, 0, FIRMWARE_HEADER_LENGTH);

    if (storedBytesLength == 0 || !Arrays.equals(storedFirmwareHeader, fileFirmwareHeader)) {
      //Can't continue upload. Erase all and start from 0
      eraseStoredData();
    } else {
      //Continuing firmware data upload from certain state.
      //Finalize transaction to get actual stored bytes size and cancel any opened prior transaction.
      finalizeTransaction();
      Logger.i("Continuing firmware upload. Stored bytes: " + storedBytesLength);
    }
  }

  private void handleEraseNotification(KDFUResponse response, byte value) {
    if (response != KDFUResponse.SUCCESS) {
      reportError("Error while erasing firmware data.");
      return;
    }
    if (value == COMMAND_DONE) {
      prepareFirstTransaction(0);
      startNextTransaction();
    }
  }

  private void handleStartTransactionNotification(KDFUResponse response) {
    if (response == KDFUResponse.SUCCESS) {
      Logger.d("Starting firmware transaction...");
      delay(new Runnable() {
        @Override
        public void run() {
          prepareTransactionChunks();
        }
      });
    }
  }

  private void handleFinalizeTransactionNotification(KDFUResponse response, byte[] storedSize) {
    int storedBytesCount = byteArrayToInt(storedSize);
    if (response == KDFUResponse.SUCCESS) {
      reportPercentProgress(storedBytesCount);
      Logger.d("Transaction finalized. Total stored bytes: " + storedBytesCount);
      if (storedBytesCount == firmwareFileBytes.length) {
        //All bytes are uploaded. Activate the firmware.
        activateFirmware();
        return;
      }
    } else {
      Logger.w("DFU transaction execution failed. Recovering...");
    }

    nextTransaction = createTransaction(firmwareFileBytes, storedBytesCount);
    delay(new Runnable() {
      @Override
      public void run() {
        startNextTransaction();
      }
    });
  }

  private void handleActivateNotification(KDFUResponse response, byte additionalInfoByte) {
    switch (response) {
      case SUCCESS:
        if (COMMAND_IN_PROGRESS == additionalInfoByte) {
          reportProgress(ACTIVATING_FIRMWARE.getPercent(), ACTIVATING_FIRMWARE.getMessage());
        } else if (COMMAND_DONE == additionalInfoByte) {
          reportProgress(FIRMWARE_ACTIVATED.getPercent(), FIRMWARE_ACTIVATED.getMessage());
          firmwareUpdateListener.onFinished(System.currentTimeMillis() - initializeTimestamp);
        }
        break;
      case IMAGE_CHECKSUM_INVALID:
        reportError("Received image checksum was invalid.");
        break;
      case IMAGE_HEADER_INVALID:
        reportError("Received image header has incorrect format.");
        break;
      case IMAGE_SIZE_INVALID:
        reportError("Received image size was invalid.");
        break;
    }
  }

  void prepareTransactionChunks() {
    BluetoothDeviceCharacteristic dataCharacteristic;
    try {
      dataCharacteristic = serviceStore.getDfuDataCharacteristic();
    } catch (ServiceAbsentException | CharacteristicAbsentException e) {
      reportError(e.getMessage());
      return;
    }

    List<byte[]> transactionChunks = nextTransaction.getChunksToSend();
    sendChunk(0, transactionChunks, dataCharacteristic);
  }

  void sendChunk(final int index, final List<byte[]> transactionChunks, final BluetoothDeviceCharacteristic dataCharacteristic) {
    if (index >= transactionChunks.size()) {
      finalizeTransaction();
      return;
    }
    dataCharacteristic.setWriteType(BluetoothGattCharacteristic.WRITE_TYPE_NO_RESPONSE);
    dataCharacteristic.setValue(transactionChunks.get(index));
    Logger.d("Sending firmware chunk: " + index + " | " + dataCharacteristic.getUuid() + " | " + Arrays.toString(dataCharacteristic.getValue()));
    if (!gattController.writeCharacteristic(dataCharacteristic)) {
      delay(new Runnable() {
        @Override
        public void run() {
          sendChunk(index, transactionChunks, dataCharacteristic);
        }
      }, OPERATION_TIMEOUT_MILLIS);
      Logger.d("Failed to write transaction chunk characteristic. Retrying...");
      return;
    }
    final int nextIndex = index + 1;
    delay(new Runnable() {
      @Override
      public void run() {
        sendChunk(nextIndex, transactionChunks, dataCharacteristic);
      }
    }, SEND_CHUNK_TIMEOUT_MILLIS);
  }

  void sendCommand(byte[] value) {
    BluetoothDeviceCharacteristic characteristic;
    try {
      characteristic = serviceStore.getDfuCommandCharacteristic();
      characteristic.setValue(value);
    } catch (ServiceAbsentException | CharacteristicAbsentException e) {
      reportError(e.getMessage());
      return;
    }

    if (!gattController.writeCharacteristic(characteristic, false)) {
      reportError("Failed to write to characteristic: " + characteristic.getName());
    } else {
      Logger.d("Sending command: " + characteristic.getUuid() + " | " + Arrays.toString(characteristic.getValue()));
    }
  }

  @Override
  public void onCharacteristicChanged(BluetoothGattCharacteristic characteristic) {
    BluetoothDeviceCharacteristic bleDeviceCharacteristic = new BluetoothDeviceCharacteristic(characteristic);
    KontaktDeviceCharacteristic kontaktDeviceCharacteristic = bleDeviceCharacteristic.getKontaktDeviceCharacteristic();

    if (kontaktDeviceCharacteristic != KontaktDeviceCharacteristic.KDFU_RESPONSE
        && kontaktDeviceCharacteristic != KontaktDeviceCharacteristic.KDFU_DATA) {
      reportError("Received unknown response.");
      return;
    }

    if (kontaktDeviceCharacteristic == KontaktDeviceCharacteristic.KDFU_DATA) {
      handleReadStateNotification(characteristic.getValue());
      return;
    }

    byte[] value = characteristic.getValue();
    KDFUCommand command = KDFUCommand.fromCode(value[0]);
    KDFUResponse response = KDFUResponse.fromCode(value[1]);
    if (command == null) {
      reportError("Unknown characteristic change command. Code: " + value[0]);
      return;
    }
    if (response == null) {
      reportError("Unknown characteristic change response. Code: " + value[1]);
      return;
    }

    switch (command) {
      case ERASE:
        byte eraseInfoByte = value[value.length - 1];
        handleEraseNotification(response, eraseInfoByte);
        break;
      case START_TRANSACTION:
        handleStartTransactionNotification(response);
        break;
      case FINALIZE_TRANSACTION:
        byte[] storedSize = Arrays.copyOfRange(value, 2, value.length);
        handleFinalizeTransactionNotification(response, storedSize);
        break;
      case ACTIVATE:
        byte activateInfoByte = value[value.length - 1];
        handleActivateNotification(response, activateInfoByte);
        break;
    }
  }

  @Override
  public void onWriteSuccess(WriteResponse response) {
    //We only write characteristic during authorization so it's safe to assume that response is always for authorization.
    authorizationService.onAuthorizationCommandWriteSuccess();
  }

  @Override
  public void onWriteFailure(Cause cause) {
    reportError(cause.name());
  }

  @Override
  public void onDescriptorWriteSuccess(BluetoothGattDescriptor descriptor) {
    BluetoothDeviceCharacteristic bleDeviceCharacteristic = new BluetoothDeviceCharacteristic(descriptor.getCharacteristic());
    KontaktDeviceCharacteristic kontaktDeviceCharacteristic = bleDeviceCharacteristic.getKontaktDeviceCharacteristic();

    if (kontaktDeviceCharacteristic == KontaktDeviceCharacteristic.KDFU_RESPONSE) {
      try {
        enableDataNotification();
      } catch (CharacteristicAbsentException | ServiceAbsentException e) {
        reportError(e.getMessage());
      }
    } else if (kontaktDeviceCharacteristic == KontaktDeviceCharacteristic.KDFU_DATA) {
      reportProgress(NOTIFICATIONS_ENABLED.getPercent(), NOTIFICATIONS_ENABLED.getMessage());
      readState();
    }
  }

  @Override
  public void onDescriptorWriteFailure(BluetoothGattDescriptor descriptor) {
    reportError("Error while writing to descriptor: " + descriptor.getUuid().toString());
  }

  @Override
  public void close() {
    handler.removeCallbacksAndMessages(null);
    gattController.close();
    firmwareFilesManager.close();
    firmwareUpdateListener = null;
  }

  private FirmwareFileCallback createFirmwareFileCallback() {
    return new FirmwareFileCallback() {
      @Override
      public void onFileAvailable(File firmwareFile) {
        try {
          firmwareFileBytes = FileUtils.toByteArray(new FileInputStream(firmwareFile));
          authorize();
        } catch (IOException e) {
          reportError(e.getMessage());
        }
      }

      @Override
      public void onError(KontaktDfuException exception) {
        reportError(exception.getMessage());
      }
    };
  }

  private DfuAuthorizationService.AuthorizationCallback createAuthorizationCallback() {
    return new DfuAuthorizationService.AuthorizationCallback() {
      @Override
      public void onAuthorized() {
        try {
          reportProgress(AUTHORIZED.getPercent(), AUTHORIZED.getMessage());
          enableDFUResponseNotification();
        } catch (CharacteristicAbsentException | ServiceAbsentException e) {
          reportError(e.getMessage());
        }
      }

      @Override
      public void onError(String message) {
        reportError(message);
      }
    };
  }

  private Transaction createTransaction(byte[] firmwareFileBytes, int storedBytesCount) {
    return new Transaction(firmwareFileBytes, storedBytesCount);
  }

  void delay(Runnable runnable) {
    delay(runnable, OPERATION_TIMEOUT_MILLIS);
  }

  void delay(Runnable runnable, int delay) {
    handler.postDelayed(runnable, delay);
  }

  void reportProgress(int percent, String message) {
    if (firmwareUpdateListener != null) {
      firmwareUpdateListener.onProgress(percent, message);
    }
  }

  void reportPercentProgress(double storedBytes) {
    double uploadPercent = (storedBytes / (double) firmwareFileBytes.length) * 100D;
    double delta = FIRMWARE_UPLOADED.getPercent() - UPLOADING_FIRMWARE.getPercent();
    double normalizedPercent = ((delta / 100D) * uploadPercent) + UPLOADING_FIRMWARE.getPercent();
    reportProgress((int) normalizedPercent, UPLOADING_FIRMWARE.getMessage());
  }

  void reportError(String message) {
    if (firmwareUpdateListener != null) {
      firmwareUpdateListener.onError(new KontaktDfuException(message));
    }
  }
}
