package com.kontakt.sdk.android.ble.dfu;

import android.util.Base64;
import com.kontakt.sdk.android.ble.connection.GattController;
import com.kontakt.sdk.android.ble.connection.KontaktDeviceServiceStore;
import com.kontakt.sdk.android.ble.exception.CharacteristicAbsentException;
import com.kontakt.sdk.android.ble.exception.ServiceAbsentException;
import com.kontakt.sdk.android.ble.spec.BluetoothDeviceCharacteristic;
import com.kontakt.sdk.android.cloud.IKontaktCloud;
import com.kontakt.sdk.android.cloud.response.CloudCallback;
import com.kontakt.sdk.android.cloud.response.CloudError;
import com.kontakt.sdk.android.cloud.response.CloudHeaders;
import com.kontakt.sdk.android.cloud.response.paginated.Configs;
import com.kontakt.sdk.android.common.profile.RemoteBluetoothDevice;

import static com.kontakt.sdk.android.common.util.SDKPreconditions.checkNotNull;

/**
 * Helper DFU class for authorizing to beacon device. Authorization is based on successfully issuing any securely encrypted command.
 */
public class DfuAuthorizationService {

  private final IKontaktCloud kontaktCloud;
  private final RemoteBluetoothDevice device;
  private final KontaktDeviceServiceStore serviceStore;
  private final GattController gattController;
  private AuthorizationCallback authorizationCallback;

  public static DfuAuthorizationService create(RemoteBluetoothDevice device, IKontaktCloud kontaktCloud, GattController gattController,
      KontaktDeviceServiceStore serviceStore) {
    return new DfuAuthorizationService(device, kontaktCloud, gattController, serviceStore);
  }

  DfuAuthorizationService(RemoteBluetoothDevice device, IKontaktCloud kontaktCloud, GattController gattController,
      KontaktDeviceServiceStore serviceStore) {
    checkNotNull(kontaktCloud);
    checkNotNull(device);
    checkNotNull(gattController);
    checkNotNull(serviceStore);
    this.serviceStore = serviceStore;
    this.gattController = gattController;
    this.kontaktCloud = kontaktCloud;
    this.device = device;
  }

  public void setAuthorizationCallback(AuthorizationCallback authorizationCallback) {
    this.authorizationCallback = authorizationCallback;
  }

  public void authorize() {
    checkNotNull(authorizationCallback, "AuthorizationCallback must be set first.");
    kontaktCloud.configs().readAll().withIds(device.getUniqueId()).execute(createCloudCallback());
  }

  private CloudCallback<Configs> createCloudCallback() {
    return new CloudCallback<Configs>() {
      @Override
      public void onSuccess(Configs response, CloudHeaders headers) {
        String command = response.getContent().get(0).getSecureRequest();
        if (command == null || command.trim().isEmpty()) {
          reportError("Authorization command is null or empty");
          return;
        }
        sendAuthorizationCommand(command);
      }

      @Override
      public void onError(CloudError error) {
        reportError("Error while downloading authorization command: " + error.getMessage());
      }
    };
  }

  void sendAuthorizationCommand(String command) {
    checkNotNull(command, "Authorization command is null");

    BluetoothDeviceCharacteristic secureWriteCharacteristic;
    try {
      secureWriteCharacteristic = serviceStore.getSecureWriteCharacteristic();
    } catch (ServiceAbsentException | CharacteristicAbsentException e) {
      reportError(e.getMessage());
      return;
    }

    byte[] decodedCommand = Base64.decode(command, Base64.DEFAULT);
    final byte[] currentValue = secureWriteCharacteristic.getValue();
    secureWriteCharacteristic.setValue(decodedCommand);
    if (!gattController.writeCharacteristic(secureWriteCharacteristic, false)) {
      secureWriteCharacteristic.setValue(currentValue);
      reportError("Failed to write to characteristic: " + secureWriteCharacteristic.getName());
    }
  }

  void onAuthorizationCommandWriteSuccess() {
    if (authorizationCallback != null) {
      authorizationCallback.onAuthorized();
    }
  }

  void reportError(String message) {
    if (authorizationCallback != null) {
      authorizationCallback.onError(message);
    }
  }

  interface AuthorizationCallback {

    void onAuthorized();

    void onError(String message);
  }
}
